%% Generic Linesearch Method - basic implementation
% Lindon Roberts, 2019

% To be implemented (make sure you understand the whole code first!):
% 1. Calculate search direction (line 48)
% 2. Backtracking Armijo linesearch (line 50)
% 3. Calculate asymptotic order of convergence (line 82)

clear, close all

% Define problem
a = 10;
objfun = @(x)scaled_quadratic(x,a); % generic objective function
% (note: scaled_quadratic.m can output 1st and 2nd derivatives)
x = [1; a]; % starting point x0
xmin = [0;0]; % true minimiser

% Details of true minimiser
[fmin, gmin, Hmin] = objfun(xmin);
kappa = cond(Hmin); % determines convergence of f(xk) for steepest descent [Theorem 6, Lecture 3]

% Solver settings
max_iterations = 800;
tol_g = 1e-5; % termination condition ||gradient|| <= tol
alpha0 = 1; % initial step length
tau = 0.5; % backtracking parameter
beta = 0.001; % for Armijo condition
nhistory = 10; % use last N iterates to check asymptotic rate

% Useful data to see progress of solver
n = numel(x);
xs = zeros(max_iterations+1, n); % iterate
fs = zeros(max_iterations+1,1); % objective value at each iteration
norm_gs = zeros(max_iterations+1,1); % ||gradient|| at each iteration

% Set initial data
xs(1,:) = x;
[f, g] = objfun(x);
fs(1,:) = f;
norm_gs(1,:) = norm(g);

% Begin iteration
k = 1;
fprintf('  k  |  f(xk)       |  ||grad|| \n');
fprintf('--------------------------------\n');
fprintf('  %i  |  %.4e  |  %.4e  \n', k-1, f, norm(g));
while k <= max_iterations && norm(g) >= tol_g
    % TODO: Calculate search direction
    s = zeros(n,1);
    % TODO: Backtracking Armijo linesearch
    alpha = alpha0;
    % Evaluate objective at new point
    x = x + alpha*s;
    [f, g] = objfun(x);
    if mod(k, 10) == 0
        fprintf('  %i  |  %.4e  |  %.4e  \n', k, f, norm(g));
    end
    % Save info
    xs(k+1,:) = x;
    fs(k+1,:) = f;
    norm_gs(k+1,:) = norm(g);
    k = k + 1;
end
fprintf('  %i  |  %.4e  |  %.4e   <- finished\n', k-1, f, norm(g));
fprintf('Finished after %g iterations\n', k-1)
% Finalise save data
xs = xs(1:k, :);
fs = fs(1:k, :);
norm_gs = norm_gs(1:k, :);
xdists = zeros(k,1);
for i=1:k
    xdists(i) = norm(xs(i,:) - xmin);
end

% Check asymptotic order of convergence of iterates
if numel(xdists) < nhistory
    asym_xdists = xdists;
else
    asym_xdists = xdists(end-nhistory:end);
end
% asym_xdists is a vector with ||xk-x*|| for last nhistory iterates
% TODO: Calculate convergence order for ||xk-x*||->0
% (hint: polyfit(xdata,ydata,1) performs linear regression)
order = 1;
fprintf('Iterates converge with order %1.2f\n', order);


%=====================================================
% Plot iterates, objective decrease, gradient decrease
% (no need to modify this, unless you want to)
%=====================================================

subplot(2,2,1);
npts = 30;
xplt = linspace(min(min(xs)), max(max(xs)), npts);
yplt = xplt;
%xplt = linspace(min(xs(:,1)),max(xs(:,1)),npts);
%yplt = linspace(min(xs(:,2)),max(xs(:,2)),npts);
[X,Y] = meshgrid(xplt,yplt);
Z = zeros(npts,npts);
for i=1:npts
    for j=1:npts
        Z(i,j) = log(objfun([X(i,j); Y(i,j)]));
    end
end
contour(X,Y,Z,20)
hold on
axis equal
plot(xs(:,1), xs(:,2), 'r.-', 'MarkerSize', 15);
xlabel('x1');
ylabel('x2');
hold off

subplot(2,2,2);
semilogy(fs-fmin, 'b-', 'Linewidth', 2);
hold on
xlabel('Iteration');
ylabel('Objective value - fmin');
rho = ((kappa-1)/(kappa+1))^2;
fprintf('rho_{SD} convergence rate <= %g (from kappa = %g)\n', rho, kappa);
semilogy(1:numel(fs), (fs(1)-fmin) * rho.^(0:1:numel(fs)-1), 'r--', 'Linewidth', 2);
grid on
hold off

subplot(2,2,3);
semilogy(norm_gs, 'b-', 'Linewidth', 2);
xlabel('Iteration');
ylabel('Norm of gradient');
grid on

subplot(2,2,4);
semilogy(xdists, 'b-', 'Linewidth', 2);
hold on
xlabel('Iteration');
ylabel('||x-x*||');
grid on
hold off
