#ifndef __Examples_hpp__
#define __Examples_hpp__

/**
 * @file Examples.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Examples for the course. 
 * @version 0.1
 * @date 2021-02-12
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "cfl/Data.hpp"
#include "cfl/Interp.hpp"
#include "cfl/Fit.hpp"
#include "cfl/AssetModel.hpp"
#include "cfl/InterestRateModel.hpp"

/**
 * @mainpage Examples for the course "Financial Computing with C++"
 */

namespace prb
{
  /** 
   * @defgroup prbData Data structures. 
   *
   * This module constructs input data curves for financial models. 
   * @{
   */

  /** 
   * Computes a part of yield curve for a number of interest rate
   * models (Hull-White, Nelson-Siegel, Svensson, Vasicek):
   * \f[
   * \Gamma(t) = \frac{1 - \exp(-\lambda (t-t_0))}{\lambda (t-t_0)}, 
   * \quad t\geq t_0.  
   * \f]
   * 
   * @param dLambda (\f$\lambda\geq 0\f$) The mean-reversion rate.
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return A part of yield curve for a number of interest rate
   * models.
   */
  cfl::Function yieldShape1(double dLambda, double dInitialTime);

  /** 
   * Computes a part of yield curve for a number of interest rate
   * models (Nelson-Siegel, Svensson):
   * \f[
   * \Gamma(t) = \frac{1 - \exp(-\lambda (t-t_0))}{\lambda (t-t_0)}
   * - \exp(-\lambda (t-t_0)), \quad t\geq t_0.  
   * \f]
   * 
   * @param dLambda (\f$\lambda\geq 0\f$) The mean-reversion
   * parameter.
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return A part of yield curve for a number of interest rate
   * models.
   */
  cfl::Function yieldShape2(double dLambda, double dInitialTime);

  /** 
   * Computes the forward exchange rate curve. The exchange rate is
   * the number of units of domestic currency per one unit of foreign
   * currency.
   * 
   * @param dSpotFX The spot exchange rate. 
   * @param rDomesticDiscount The domestic discount curve. 
   * @param rForeignDiscount The foreign discount curve. 
   * 
   * @return The forward exchange rate curve. 
   */
  cfl::Function forwardFX(double dSpotFX, const cfl::Function &rDomesticDiscount,
                          const cfl::Function &rForeignDiscount);
  /** 
   * Computes the curve of forward prices ("clean" or "dirty") for an
   * annuity. The annuity pays coupons \f$q \delta t\f$ at times
   * \f$(t_i)_{i=1,\dots,M}\f$ such that 
   * \f[
   *     t_0 < t_1 \leq t_0+\delta t, \quad t_{i+1}-t_i= \delta t, \quad
   *     t_M = T.
   * \f]
   * The buyer pays forward price \f$F(t)\f$ at delivery time \f$t\f$
   * and then receives coupons \f$q\delta t\f$ at payments times
   * \f$t_i>t\f$.
   * 
   * @param dRate (\f$q\f$) The annuity rate. 
   * @param dPeriod (\f$\delta t\f$) The time interval between payments. 
   * @param dMaturity (\f$T\f$) The maturity. 
   * @param rDiscount The discount curve. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * @param bClean This parameter controls the output. If its value
   * is \p true, then we compute "clean" prices. Otherwise, we
   * compute "dirty" prices that include the accrued interest from
   * the next coupon. 
   * 
   * @return The forward prices ("clean" or "dirty") for an annuity. 
   */
  cfl::Function forwardAnnuity(double dRate, double dPeriod, double dMaturity,
                               const cfl::Function &rDiscount,
                               double dInitialTime, bool bClean);
  /** @} */  

  /** 
   * @defgroup prbInterp Interpolation of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using interpolation.   
   * @{
   */

  /**
   * Computes the discount curve by the log-linear interpolation of a
   * given collection of market discount factors.
   * 
   * @param rDiscountTimes The maturities of the market discount
   * factors.
   * @param rDiscountFactors The market discount factors. 
   * @param dInitialTime The initial time. 
   * 
   * @return The discount curve obtained from the market discount
   * factors by the log-linear interpolation.
   */
  cfl::Function 
  discountLogLinInterp(const std::vector<double> & rDiscountTimes, 
		       const std::vector<double> & rDiscountFactors,
		       double dInitialTime);

  /**
   * Computes forward curve 
   * \f[
   *   F(t) = S_0 e^{q(t)(t-t_0)}, \quad t\in [t_0,t_M], 
   * \f]
   * where the cost-of-carry rate \f$q=q(t)\f$ is obtained by the
   * interpolation of the market cost-of-carry rates with method \p
   * rInterp. 
   * 
   * @param dSpot (\f$S_0\f$) The spot price of the stock. 
   * @param rDeliveryTimes \f$(t_i)_{i=1,\dots,M}\f$ The delivery
   * times for the market forward prices.
   * @param rForwardPrices The market forward prices.  
   * @param dInitialCarryRate (\f$q(t_0)\f$) The initial cost-of-carry rate.
   * @param dInitialTime (\f$t_0\f$) The initial time.
   * @param rInterp The interpolation method for the cost-of-carry rates. 
   * 
   * @return The forward curve obtained by interpolation of
   * the market cost-of-carry rates with method \p rInterp.
   */
  cfl::Function forwardCarryInterp(double dSpot, 
				   const std::vector<double> & rDeliveryTimes, 
				   const std::vector<double> & rForwardPrices,
				   double dInitialCarryRate, double dInitialTime,
				   const cfl::Interp & rInterp);

  /** @} */  

  /** 
   * @defgroup prbFit Least-square fitting of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using least-square fitting.   
   * @{
   */

  /**
   * Returns  discount curve 
   * \f[
   *    d(t) = \exp(-\gamma(t)(t-t_0)), \quad t\geq t_0, 
   * \f]
   * by the least-squares fit of yield curve \f$\gamma=\gamma(t)\f$ to
   * continuously compounded market yields with fitting method \p
   * rFit.
   * 
   * @param rTime The maturities of market discount factors.
   * @param rDF The market discount factors. 
   * @param dInitialTime  (\f$t_0\f$) The initial time.
   * @param rFit The fitting method. 
   * @param rErr The error function of the fit for discount factors.  
   *
   * @return The fitted discount curve and its error function. We also
   * obtain the fitted coefficients and their covariance matrix
   * through \p rFit.
   */
  cfl::Function
  discountYieldFit(const std::vector<double> &rTime, const std::vector<double> &rDF,
                   double dInitialTime, cfl::Fit &rFit, cfl::Function &rErr);

  /**
   * Returns the discount curve
   * \f[
   *    d(t) = \exp(-\gamma(t)(t-t_0)), \quad t\geq t_0, 
   * \f]
   * by the least-squares fit of Nelson-Siegel yield curve
   * \f[
   *  \gamma(t) = c_0 + c_1 \frac{1-e^{-\lambda(t-t_0)}}{\lambda (t-t_0)} +
   *  c_2 \left(\frac{1-e^{-\lambda(t-t_0)}}{\lambda (t-t_0)}  -
   *     e^{-\lambda (t-t_0)}\right), \quad t\geq t_0,  
   * \f]
   * to continuously compounded market yields. Here \f$c_0\f$,
   * \f$c_1\f$, and \f$c_2\f$ are the fitting constants.
   * 
   * @param rTimes The maturities of market discount factors.
   * @param rDF The market discount factors. 
   * @param dLambda (\f$\lambda\geq 0\f$) The mean-reversion rate. 
   * @param dInitialTime (\f$t_0\f$) The initial time.
   * @param rErr The error function of the fit for discount factors. 
   * @param rParam The fitted coefficients, their covariance matrix, and
   * the total \f$\chi^2\f$ fitting error.
   * 
   * @return The fitted discount curve and its error function. We also
   * obtain the fitted coefficients and their covariance matrix.
   */
  cfl::Function
  discountNelsonSiegelFit(const std::vector<double> &rTimes,
                          const std::vector<double> &rDF,
                          double dLambda, double dInitialTime,
                          cfl::Function &rErr, cfl::FitParam & rParam);

  /** @} */

  /**
   * @defgroup cflAssetOptions Options on a single stock. 
   * 
   * This module deals with valuation of options on a single stock.
   * @{
   */

  /** 
   * Computes the value of the <strong>European put option</strong>.
   * In this contract, at maturity \p dMaturity a holder of the option
   * can sell the stock at strike \p dStrike.
   * 
   * @param dStrike The strike of the option. 
   * @param dMaturity The maturity of the option.  
   * @param rModel The reference to an implementation of cfl::AssetModel. 
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.  
   */
  cfl::MultiFunction put(double dStrike, double dMaturity, cfl::AssetModel &rModel);

  /** 
   * Computes the value of the <strong>American put option</strong>.
   * In this contract, at any exercise time (from \p rExerciseTimes) a
   * holder of the option can sell the stock at strike \p dStrike.
   * 
   * @param dStrike The strike of the option.
   * @param rExerciseTimes The vector of exercise times. The first exercise
   * time is strictly greater than the initial time. 
   * @param rModel The reference to the implementation of cfl::AssetModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model. 
   */
  cfl::MultiFunction
  americanPut(double dStrike,
	      const std::vector<double> &rExerciseTimes,
	      cfl::AssetModel &rModel);

  /** 
   * Computes the value of the <strong>up-or-down-and-out barrier
   * option</strong>. The payoff of the option at maturity (last
   * barrier time) is notional amount \p dNotional if the stock price
   * has not crossed the lower and upper barriers for all barrier
   * times. Otherwise, the option expires worthless.
   * 
   * @param dLowerBarrier The lower barrier. 
   * @param dUpperBarrier The upper barrier. 
   * @param dNotional The notional amount. 
   * @param rBarrierTimes The vector of barrier times. The first
   * time is greater than the initial time. The last barrier time is
   * the maturity of the option. 
   * @param rModel The reference to an implementation of cfl::AssetModel. 
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.  
   */
  cfl::MultiFunction
  barrierUpDownOut(double dNotional, double dLowerBarrier,
		   double dUpperBarrier,
		   const std::vector<double> &rBarrierTimes,
		   cfl::AssetModel &rModel);

  /**
   * Computes the value of the <strong>down-and-out American
   * call</strong>.  The option behaves as the American call option
   * with strike \p dStrike and exercise times \p rExerciseTimes until
   * the first barrier time when the stock price hits lower barrier \p
   * dBarrier. At this exit time the option is canceled.
   * 
   * @param dBarrier The lower barrier (\p dBarrier < \p dStrike). 
   * @param rBarrierTimes The vector of barrier times. The first
   * time is greater than the initial time. 
   * @param dStrike The strike of the option. 
   * @param rExerciseTimes The vector of exercise times. The first exercise
   * time is strictly greater than the initial time.  
   * @param rModel The reference to an implementation of cfl::AssetModel. 
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.  
   */
  cfl::MultiFunction
  downOutAmericanCall(double dBarrier,
		      const std::vector<double> &rBarrierTimes,
		      double dStrike,
		      const std::vector<double> &rExerciseTimes,
		      cfl::AssetModel &rModel);

  /** 
   * Computes the value of the <strong>swing option</strong>.  The
   * holder can exercise the option \p iNumberOfExercises times.  At
   * each exercise time he can buy only one stock for strike \p
   * dStrike.
   * 
   * @param dStrike The strike of the option.
   * @param rExerciseTimes The vector of exercise times. 
   * @param iNumberOfExercises The maximal number of exercises. 
   * @param rModel The reference to an implementation of cfl::AssetModel. 
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.  
   */

  cfl::MultiFunction swing(double dStrike,
			   const std::vector<double> &rExerciseTimes,
			   unsigned iNumberOfExercises, cfl::AssetModel &rModel);

  /** @} */

  /**
   * @defgroup cflInterestRateOptions Options on interest rates. 
   * 
   * This module deals with valuation of standard and barrier 
   * options on interest rates.
   * 
   * @{
   */

  /** 
   * Computes the price of <strong>interest rate cap</strong>. 
   * 
   * @param rCap The parameters of interest rate cap. 
   * @param rModel Reference to implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction cap(const cfl::Data::CashFlow &rCap, cfl::InterestRateModel &rModel);

  /** 
   * Compute the present value of <strong>interest rate
   * swap</strong>. The computations are done at issue time of the
   * swap contract. 
   * 
   * @param rSwap The parameters of the underlying interest rate
   * swap. 
   * @param rModel Reference to implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction swap(const cfl::Data::Swap &rSwap, cfl::InterestRateModel &rModel);

  /** 
   * Computes the price of <strong>interest rate swaption</strong>.
   * In this contract, at maturity \p dMaturity a holder of the option
   * has the right to enter the interest rate swap with issue time \p
   * dMaturity and parameters \p rSwap.
   * 
   * @param rSwap The parameters of the underlying interest rate
   * swap. 
   * @param dMaturity The maturity of the option. This time is also
   * the issue time for the underlying swap contract. 
   * @param rModel The reference to the implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction swaption(const cfl::Data::Swap &rSwap, double dMaturity,
			      cfl::InterestRateModel &rModel);

  /** 
   * Computes the price of <strong>cancellable collar</strong>. 
   * In this contract, holder of the option can terminate the
   * underlying interest rate collar at any payment time.
   * Note that in the case of termination the current payment 
   * is still taking place. 
   * 
   * @param rCap The parameters of the underlying interest rate
   * collar. Here <code>rCap.rate</code> defines the cap rate. 
   * @param dFloorRate The floor rate in the underlying interest
   * rate collar. 
   * @param rModel The reference to the implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction cancellableCollar(const cfl::Data::CashFlow &rCap,
				       double dFloorRate,
				       cfl::InterestRateModel &rModel);
  /** 
   * Computes the price of <strong>down-and-out cap</strong>.  The
   * down-and-out cap generates the same cash flow as a standard
   * interest rate cap up to and including the payment time when the
   * float rate hits lower barrier \p dLowerBarrier. After that, the
   * cap is terminated.
   * 
   * @param rCap The parameters of the underlying interest rate
   * cap. 
   * @param dLowerBarrier The lower barrier for float rate. 
   * @param rModel The reference to the implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction downOutCap(const cfl::Data::CashFlow &rCap,
				double dLowerBarrier,
				cfl::InterestRateModel &rModel);

  /** 
   * Computes  <strong>futures price on LIBOR</strong>.
   * The set of times, when the futures price is evaluated is given by 
   * \f[
   * t_i = t_0 + i\frac{T-t_0}{M}, \quad i=1,\dots,M, 
   * \f]
   * where \f$t_0=\f$ \p rModel.initialTime(), \f$T=\f$ \p
   * dFuturesMaturity, and \f$M=\f$ \p iFuturesTimes.
   * At maturity \p dFuturesMaturity, the settlement price is 
   * \f[
   * 1 - L(T,T+\Delta), 
   * \f]
   * where \f$\Delta = \f$ \p dLiborPeriod and \f$L(s,t)\f$ is the
   * LIBOR computed at \f$s\f$ for maturity \f$t\f$.
   * 
   * @param dLiborPeriod The LIBOR period in the futures. 
   * @param iFuturesTimes The number of futures times. 
   * @param dMaturity The maturity. 
   * @param rModel The reference to the implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction futuresOnLibor(double dLiborPeriod,
                                    unsigned iFuturesTimes, double dMaturity,
                                    cfl::InterestRateModel &rModel);
  /** @} */  
}

#endif // of __Examples_hpp__
