#include "test/Main.hpp"
#include "test/Data.hpp"
#include "test/Print.hpp"
#include "test/Black.hpp"
#include "test/HullWhite.hpp"
#include "Examples/Output.hpp"
#include "Examples/Examples.hpp"

using namespace test;
using namespace cfl;
using namespace std;
using namespace test::Data;

// DATA CURVES FOR FINANCIAL MODELS

void yieldShape1()
{
  test::print("YIELD SHAPE 1");

  double dLambda = 0.05;
  double dInitialTime = 2.;

  print(dLambda, "lambda");
  print(dInitialTime, "initial time", true);
  Function uYield = prb::yieldShape1(dLambda, dInitialTime);
  double dInterval = 4.75;
  test::Data::print(uYield, dInitialTime + 0.001, dInterval);
}

void yieldShape2()
{
  test::print("YIELD SHAPE 2");

  double dLambda = 0.05;
  double dInitialTime = 2.;

  print(dLambda, "lambda");
  print(dInitialTime, "initial time", true);
  Function uYield = prb::yieldShape2(dLambda, dInitialTime);
  double dInterval = 4.75;
  test::Data::print(uYield, dInitialTime + 0.001, dInterval);
}

void forwardFX()
{
  test::print("FORWARD PRICES FOR EXCHANGE RATES");

  double dSpotFX = 100;
  double dDom = 0.12;
  double dFor = 0.05;
  double dInitialTime = 1.;

  print(dInitialTime, "initial time");
  print(dSpotFX, "spot FX rate");
  print(dDom, "domestic interest rate");
  print(dFor, "foreign interest rate", true);

  Function uDomestic = cfl::Data::discount(dDom, dInitialTime);
  Function uForeign = cfl::Data::discount(dFor, dInitialTime);
  Function uForwardFX = prb::forwardFX(dSpotFX, uDomestic, uForeign);
  double dInterval = 0.5;
  test::Data::print(uForwardFX, dInitialTime, dInterval);
}

void forwardAnnuity()
{
  test::print("FORWARD PRICES FOR AN ANNUITY");

  cfl::Data::CashFlow uAnnuity = test::HullWhite::swapParameters();
  uAnnuity.notional = 1;
  double dRate = uAnnuity.rate;
  double dInitialTime = 1.;
  Function uDiscount = cfl::Data::discount(dRate, dInitialTime);

  print(dRate, "interest rate");
  print(dInitialTime, "initial time", true);
  test::printCashFlow(uAnnuity, "annuity parameters");

  for (int iI = 0; iI < 2; iI++)
    {
      bool bClean = (iI == 0) ? true : false;
      if (bClean)
	{
	  cout << "clean prices:" << endl;
	}
      else
	{
	  cout << "dirty prices:" << endl;
	}
      cout << endl;
      double dRate = uAnnuity.rate;
      double dPeriod = uAnnuity.period;
      double dMaturity = dInitialTime + dPeriod * uAnnuity.numberOfPayments;
      Function uForwardAnnuity =
        prb::forwardAnnuity(dRate, dPeriod, dMaturity, uDiscount,
                            dInitialTime, bClean);
      double dInterval =
        uAnnuity.period * uAnnuity.numberOfPayments / 1.1;
      test::Data::print(uForwardAnnuity, dInitialTime, dInterval);
    }
}

// INTERPOLATION OF DATA CURVES

void discountLogLinInterp()
{
  test::print("LOG LINEAR INTERPOLATION OF DISCOUNT CURVE");

  double dInitialTime = 1.;

  auto uDF = test::Data::getDiscount(dInitialTime);
  
  Function uDiscount =
    prb::discountLogLinInterp(uDF.first, uDF.second, dInitialTime);

  double dInterval = uDF.first.back() - dInitialTime;
  test::Data::print(uDiscount, dInitialTime, dInterval);
}

void forwardCarryInterp()
{
  test::print("FORWARD PRICES BY INTERPOLATION OF COST-OF-CARRY RATES");

  double dSpot = 100;
  double dInitialTime = 1.;

  auto uF = test::Data::getForward(dSpot, dInitialTime);

  double dInitialCarryRate = std::log(uF.second.front()/dSpot)/(uF.first.front()-dInitialTime);
  print(dInitialCarryRate, "initial carry rate", true);

  Function uResult =
    prb::forwardCarryInterp(dSpot, uF.first, uF.second,
			    dInitialCarryRate, dInitialTime,
			    cfl::NInterp::cspline());

  print("interpolation with cubic spline:");
  double dInterval = uF.first.back() - dInitialTime;
  test::Data::print(uResult, dInitialTime, dInterval);
}

// LEAST-SQUARES FITTING OF DATA CURVES

const std::string c_sDF("Fitted discount factors and their errors:");
const std::string c_sConstYield("We fit with constant yield.");

void discountYieldFit()
{
  test::print("DISCOUNT CURVE OBTAINED BY LEAST-SQUARES FIT OF YIELD CURVE");

  double dInitialTime = 1.;

  auto uDF = test::Data::getDiscount(dInitialTime);

  print(c_sConstYield);
  cfl::Fit uFit = cfl::NFit::linear(cfl::Function(1.));

  Function uErr;
  Function uDiscount =
    prb::discountYieldFit(uDF.first, uDF.second,
			  dInitialTime, uFit, uErr);
                            
  double dInterval = uDF.first.back() - dInitialTime;
  test::Data::printFit(uFit.param());
  test::Data::print(c_sDF, uDiscount, uErr, dInitialTime, dInterval);
}

void discountNelsonSiegelFit()
{
  test::print("LEAST-SQUARES FIT OF DISCOUNT CURVE IN NELSON-SIEGEL MODEL");

  double dLambda = 0.05;
  double dInitialTime = 1.;

  print(dLambda, "lambda");
  auto uDF = test::Data::getDiscount(dInitialTime);

  Function uErr;
  FitParam uParam;
  Function uDiscount =
    prb::discountNelsonSiegelFit(uDF.first, uDF.second,
				 dLambda, dInitialTime,
				 uErr, uParam);
  double dInterval = uDF.first.back() - dInitialTime;
  test::Data::printFit(uParam);
  test::Data::print(c_sDF, uDiscount, uErr, dInitialTime, dInterval);
}

// OPTIONS ON A SINGLE STOCK IN BLACK MODEL

MultiFunction put(AssetModel &rModel)
{
  test::print("EUROPEAN PUT OPTION IN ASSET MODEL");

  double dStrike = test::Black::c_dSpot;
  double dMaturity = test::Black::c_dMaturity;

  print(dStrike, "strike");
  print(dMaturity, "maturity", true);

  return prb::put(dStrike, dMaturity, rModel);
}

MultiFunction americanPut(AssetModel &rModel)
{
  test::print("AMERICAN PUT OPTION IN ASSET MODEL");

  double dStrike = test::Black::c_dSpot;
  const std::vector<double> uExerciseTimes = test::Black::exerciseTimes();

  print(dStrike, "strike", true);
  test::print(uExerciseTimes.begin(), uExerciseTimes.end(), "exercise times");

  return prb::americanPut(dStrike, uExerciseTimes, rModel);
}

MultiFunction barrierUpDownOut(AssetModel &rModel)
{
  test::print("BARRIER UP-OR-DOWN-AND-OUT OPTION IN ASSET MODEL");

  double dLowerBarrier = test::Black::c_dSpot * 0.9;
  double dUpperBarrier = test::Black::c_dSpot * 1.1;
  double dNotional = test::Black::c_dNotional;
  const std::vector<double> uBarrierTimes = test::Black::barrierTimes();

  print(dLowerBarrier, "lower barrier");
  print(dUpperBarrier, "upper barrier");
  print(dNotional, "notional", true);
  test::print(uBarrierTimes.begin(), uBarrierTimes.end(), "barrier times");

  return prb::barrierUpDownOut(dNotional, dLowerBarrier,
                               dUpperBarrier, uBarrierTimes,
                               rModel);
}

MultiFunction
downOutAmericanCall(AssetModel &rModel)
{
  test::print("DOWN-AND-OUT AMERICAN CALL OPTION IN ASSET MODEL");

  double dStrike = test::Black::c_dSpot;
  const std::vector<double>
    uExerciseTimes = test::Black::exerciseTimes();
  double dLowerBarrier = test::Black::c_dSpot * 0.9;
  const std::vector<double>
    uBarrierTimes = test::Black::barrierTimes();

  print(dStrike, "strike");
  print(dLowerBarrier, "lower barrier", true);
  test::print(uExerciseTimes.begin(), uExerciseTimes.end(), "exercise times");
  test::print(uBarrierTimes.begin(), uBarrierTimes.end(), "barrier times");

  return prb::downOutAmericanCall(dLowerBarrier,
                                  uBarrierTimes, dStrike,
                                  uExerciseTimes, rModel);
}

MultiFunction swing(AssetModel &rModel)
{
  test::print("SWING OPTION IN ASSET MODEL");

  double dStrike = test::Black::c_dSpot;
  const std::vector<double> uExerciseTimes = test::Black::exerciseTimes();
  unsigned iNumberOfExercises = uExerciseTimes.size() / 3;

  print(dStrike, "strike");
  print(iNumberOfExercises, "maximal number of exercises", true);
  test::print(uExerciseTimes.begin(), uExerciseTimes.end(), "exercise times");

  return prb::swing(dStrike, uExerciseTimes,
                    iNumberOfExercises, rModel);
}


// INTEREST RATE OPTIONS IN HULL-WHITE MODEL

cfl::MultiFunction cap(InterestRateModel &rModel)
{
  test::print("CAP IN INTEREST RATE MODEL");

  cfl::Data::CashFlow uCapParameters = test::HullWhite::swapParameters();
  uCapParameters.rate = test::HullWhite::c_dYield * 1.1;

  test::printCashFlow(uCapParameters, "cap parameters");

  return prb::cap(uCapParameters, rModel);
}

cfl::MultiFunction swap(InterestRateModel &rModel, bool bPayFloat)
{
  test::print("SWAP IN INTEREST RATE MODEL");

  cfl::Data::Swap uSwap = test::HullWhite::swapParameters();
  uSwap.payFloat = bPayFloat;

  test::printSwap(uSwap, "swap parameters");

  return prb::swap(uSwap, rModel);
}

cfl::MultiFunction swaption(InterestRateModel & rModel, bool bPayFloat)
{
  test::print("SWAPTION IN INTEREST RATE MODEL");

  cfl::Data::Swap uSwap = test::HullWhite::swapParameters();
  uSwap.payFloat = bPayFloat;
  double dMaturity = test::HullWhite::c_dMaturity;

  test::printSwap(uSwap, "swap parameters");
  print(dMaturity, "maturity", true);

  return prb::swaption(uSwap, dMaturity, rModel);
}

cfl::MultiFunction cancellableCollar(InterestRateModel &rModel)
{
  test::print("CANCELLABLE COLLAR IN INTEREST RATE MODEL");

  cfl::Data::CashFlow uCapParameters = test::HullWhite::swapParameters();
  uCapParameters.rate = test::HullWhite::c_dYield * 1.1;
  double dFloorRate = test::HullWhite::c_dYield * 0.9;

  test::printCashFlow(uCapParameters, "cap parameters");
  print(dFloorRate, "floor rate", true);

  return prb::cancellableCollar(uCapParameters, dFloorRate,
                                rModel);
}

cfl::MultiFunction downOutCap(InterestRateModel &rModel)
{
  test::print("DOWN-AND-OUT CAP IN INTEREST RATE MODEL");

  cfl::Data::CashFlow uCapParameters = test::HullWhite::swapParameters();
  uCapParameters.rate = test::HullWhite::c_dYield * 1.1;
  double dLowerBarrier = test::HullWhite::c_dYield * 0.9;

  test::printCashFlow(uCapParameters, "cap parameters");
  print(dLowerBarrier, "lower barrier", true);

  return prb::downOutCap(uCapParameters, dLowerBarrier,
                         rModel);
}

cfl::MultiFunction futuresOnLibor(InterestRateModel &rModel)
{
  test::print("LIBOR FUTURES IN INTEREST RATE MODEL");

  double dLiborPeriod = test::HullWhite::c_dPeriod;
  unsigned iFuturesTimes = 20;
  double dMaturity = 0.25;

  print(dLiborPeriod, "period for LIBOR");
  print(iFuturesTimes, "number of futures times");
  print(dMaturity, "maturity", true);

  return prb::
    futuresOnLibor(dLiborPeriod, iFuturesTimes, dMaturity, rModel);
}

std::function<void()> test_Examples()
{
  return []() {
	   print("DATA CURVES FOR FINANCIAL MODELS");
    
	   yieldShape1();
	   yieldShape2();
	   forwardFX();
	   forwardAnnuity();

	   print("INTERPOLATION OF DATA CURVES");

	   discountLogLinInterp();
	   forwardCarryInterp();

	   print("LEAST-SQUARES FITTING OF DATA CURVES");

	   discountYieldFit();
	   discountNelsonSiegelFit();
    
	   print("OPTIONS ON A SINGLE STOCK IN BLACK MODEL");

	   AssetModel uBlack = test::Black::model();
	   test::Black::report(put,uBlack);
	   test::Black::report(americanPut,uBlack);
	   test::Black::report(barrierUpDownOut,uBlack);
	   test::Black::report(downOutAmericanCall,uBlack);
	   test::Black::report(swing,uBlack);
	   
	   print("INTEREST RATE OPTIONS IN HULL-WHITE MODEL");

	   InterestRateModel uHullWhite = test::HullWhite::model();
	   test::HullWhite::report(cap, uHullWhite);
	   test::HullWhite::report(swap, uHullWhite);
	   test::HullWhite::report(swaption,uHullWhite);
	   test::HullWhite::report(cancellableCollar,uHullWhite);
	   test::HullWhite::report(downOutCap,uHullWhite);
	   test::HullWhite::report(futuresOnLibor,uHullWhite);
	 };
}

int main()
{
  project(test_Examples(), PROJECT_NAME, PROJECT_NAME,
          "Examples");
}
