#ifndef __cflBlackModel_hpp__
#define __cflBlackModel_hpp__

/**
 * @file BlackModel.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Implementation of Black model. 
 * @version 1.0
 * @date 2021-01-12
 * 
 * @copyright Copyright (c) 2020
 * 
 */

#include "cfl/Brownian.hpp"
#include "cfl/AssetModel.hpp"

namespace cfl
{
  /** 
   * @ingroup cflAssetModel
   *
   * @defgroup cflBlack Black model for a single asset.
   * 
   * This module contains implementation of the classical
   * Black model for a single asset. 
   * 
   * @see cfl::AssetModel and cfl::Brownian
   * @{
   */

  /** 
   * @brief  Black model for a single asset. 
   * 
   * This namespace contains functions and classes related with the 
   * classical Black model for a single asset.
   */
  namespace Black
  {
    /** 
     * @brief  The parameters of Black model. 
     * 
     * This class defines the parameters of the Black model. The set of parameters 
     * consists of discount, forward, shape, and volatility curves. 
     */
    class Data
    {
    public:
      /**
       * Default constructor. 
       */
      Data(){};

      /** 
       * Constructs the parameters of classical Black model.
       * 
       * @param rDiscount A discount curve.
       * @param rForward A forward curve.
       * @param rVolatility A volatility curve. 
       * @param dInitialTime Initial time given as year fraction.
       */
      Data(const Function &rDiscount, const Function &rForward,
           const Function &rVolatility, double dInitialTime);

      /** 
       * Constructs the parameters of classical Black model with constant volatility.
       * 
       * @param rDiscount A discount curve.
       * @param rForward A forward curve.
       * @param dSigma A volatility. 
       * @param dInitialTime Initial time given as year fraction.
       */
      Data(const Function &rDiscount, const Function &rForward,
           double dSigma, double dInitialTime);

      /** 
       * Constructs the parameters of general Black model.
       * 
       * @param rDiscount A discount curve.
       * @param rForward A forward curve.
       * @param rVolatility A volatility curve. 
       * @param rShape A shape function. This function defines the shape of 
       * movements of the curve of forward prices. 
       * @param dInitialTime Initial time given as year fraction.
       */
      Data(const Function &rDiscount, const Function &rForward,
           const Function &rVolatility, const Function &rShape,
           double dInitialTime);

      /** 
       * Constructs the parameters of general Black model with stationary volatility.
       * 
       * @param rDiscount A discount curve.
       * @param rForward A forward curve.
       * @param dSigma The volatility of spot price. 
       * @param dLambda The mean-reversion coefficient for the log of spot price under 
       * the risk-neutral measure. 
       * @param dInitialTime Initial time given as year fraction.
       */
      Data(const Function &rDiscount, const Function &rForward,
           double dSigma, double dLambda, double dInitialTime);

      /**
       * @copydoc HullWhite::Data::initialTime 
       */
      double initialTime() const;

      /**
       * @copydoc HullWhite::Data::discount
       */
      const Function &discount() const;

      /**
       * Accessor function to forward curve. 
       * 
       * @return Forward curve for financial asset. 
       */
      const Function &forward() const;

      /**
       * @copydoc HullWhite::Data::volatility
       */
      const Function &volatility() const;

      /** 
       * Accessor function to "shape" curve.  
       * 
       * @return The shape function. This function defines the shape of 
       * changes in forward prices.  
       */
      const Function &shape() const;

    private:
      Function m_uDiscount;
      Function m_uForward;
      Function m_uVol;
      Function m_uShape;
      double m_dInitialTime;
    };

    /** 
     * Implements AssetModel as Black model. 
     * 
     * @param rData The parameters of Black model. 
     * @param dInterval The interval of initial values for the relative 
     * change in the spot price of the underlying asset. 
     * @param dQuality The trade-off between speed and accuracy. 
     * @return Implementation of AssetModel  as Black model. 
     */
    AssetModel model(const Data &rData, double dInterval, double dQuality);

    /** 
     * Implements AssetModel as Black model.
     * 
     * @param rData The parameters of Black model. 
     * @param dInterval The interval of initial values for the relative 
     * change in the spot price of the underlying asset.
     * @param rBrownian Implementation of class Brownian. 
     * @return Implementation of AssetModel  as Black model. 
     */
    AssetModel model(const Data &rData, double dInterval, const Brownian &rBrownian);
  } // namespace Black
  /** @} */
} // namespace cfl

#include "cfl/Inline/iBlackModel.hpp"
#endif // of __cflBlackModel_hpp__
