#ifndef __cflBrownian_hpp__
#define __cflBrownian_hpp__

/**
 * @file Brownian.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Financial model driven by Brownian motion.
 * @version 1.0
 * @date 2021-01-12
 * 
 * @copyright Copyright (c) 2020
 * 
 */

#include "Slice.hpp"
#include "GaussRollback.hpp"
#include "Interp.hpp"
#include "Ind.hpp"

namespace cfl
{
  /** 
   * @ingroup cflCommonElements
   *
   * @defgroup cflBrownian Basic model with Brownian motion. 
   *
   * This module contains implementation of the basic financial 
   * model where the state process is a one-dimensional Brownian motion.
   * @{
   */

  /** 
   * @brief Interface class for the basic financial model with
   * Brownian motion.
   *
   * This is the interface class for the financial model with zero
   * interest rate, where the state process is a one-dimensional
   * Brownian motion.  This abstract class is used to implement
   * concrete class Brownian.
   * 
   * @see Brownian and NBrownian
   */
  class IBrownian
  {
  public:
    /**
     * Virtual destructor. 
     */
    virtual ~IBrownian(){};

    /** 
     * Virtual constructor. 
     * 
     * @param rVar The vector of variances of the Brownian motion.  \p
     * rVar[i] defines the average variance between \p rEventTimes[0]
     * and \p rEventTimes[i]. This vector has the same size as \p
     * rEventTimes.
     * @param rEventTimes The vector of event times in the model. 
     * @param dInterval The width of the interval of initial values of
     * the underlying Brownian motion.
     * @return A pointer to new implementation of IBrownian. This
     * implementation has the parameters \p rEventTimes, \p rVar, and
     * \p dInterval.
     */
    virtual IBrownian *newModel(const std::vector<double> &rVar,
                                const std::vector<double> &rEventTimes,
                                double dInterval) const = 0;

    /**
     * @copydoc IInterestRateModel::model
     */
    virtual const IModel &model() const = 0;
  };

  /** 
   * @brief  Concrete class for the basic financial model with Brownian motion. 
   *
   * This is the standard concrete class for the basic financial model
   * where the interest rate equals to zero and the state process is
   * given by a one-dimensional Brownian motion. This class is
   * implemented by a new implementation of interface class IBrownian. 
   * 
   * @see IBrownian and NBrownian
   */
  class Brownian
  {
  public:
    /** 
     * Constructs \p *this from new implementation of IBrownian. 
     *
     *@param pNewP A pointer to new implementation of IBrownian.
     */
    explicit Brownian(IBrownian *pNewP);

    /** 
     * Changes event times, variances, and the interval of initial
     * values for \p *this. After this operation, the member function
     * model() will return a reference to a standard implementation of
     * the model.
     * 
     * @param rVar The vector of variances of the Brownian motion.  \p
     * rVar[i] defines the average variance between \p rEventTimes[0]
     * and \p rEventTimes[i]. This vector has the same size as \p
     * rEventTimes.
     * @param rEventTimes The vector of event times in the model. 
     * @param dInterval The width of the interval of initial values of
     * the underlying Brownian motion.
     */
    void assign(const std::vector<double> &rVar,
                const std::vector<double> &rEventTimes,
                double dInterval);

    /**
     * @copydoc InterestRateModel::eventTimes
     */
    const std::vector<double> &eventTimes() const;

    /**
    * Returns the value of the Brownian motion at the event time with index 
    * \p iEventTime. The same as <code>model().state(iEventTime,0)</code>. 
    * 
    * @param iEventTime The index of event time. 
    * @return The value of the Brownian motion at the given event time. 
    */
    Slice state(unsigned iEventTime) const;

    /**
     * @copydoc InterestRateModel::model
     */
    const IModel &model() const;

  private:
    std::shared_ptr<IBrownian> m_pModel;
  };

  /** 
   * @brief  Implementations of class Brownian.
   * 
   * This namespace contains implementations of the basic financial
   * model where the state process is a one-dimensional Brownian motion and
   * the interest rate equals zero.
   *
   * @see IBrownian and Brownian
   */
  namespace NBrownian
  {
    /** 
     * Implements Brownian model from different components. The storage is 
     * in the form of symmetric equally spaced grid with \f$ 2^n \f$ elements 
     * (to facilitate the use of radix-2 Fast Fourier Transform).
     * 
     * @param dQuality A trade-off between speed and accuracy of the implementation 
     * of the basic state process. 
     * @param rRollback An implementation of the operator of conditional expectation with 
     * respect to gaussian distribution. 
     * @param rInd A numerically efficient implementation of discontinuous functions. 
     * @param rInterp An implementation of numerical interpolation. 
     * @return Implementation of Brownian model. 
     */
    Brownian model(double dQuality,
                   const GaussRollback &rRollback = cfl::NGaussRollback::chain(),
                   const Ind &rInd = cfl::NInd::linear(),
                   const Interp &rInterp = cfl::NInterp::cspline());

  } // namespace NBrownian
  /** @} */
} // namespace cfl

#include "cfl/Inline/iBrownian.hpp"
#endif // of __cflBrownian_hpp__
