#ifndef __cflInterestRateModel_hpp__
#define __cflInterestRateModel_hpp__

/**
 * @file InterestRateModel.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Financial models for Interest rates. 
 * @version 1.0
 * @date 2021-01-12
 * 
 * @copyright Copyright (c) 2020
 * 
 */

#include <cfl/Slice.hpp>

namespace cfl
{
  /**
   * @ingroup cflModel
   * @defgroup cflInterestRateModel Interest rate models. 
   * This module deals with interest rate models. 
   * @{
   */

  /** 
   * @brief  Interface class for interest rate model. 
   * 
   * This is the abstract class for interest rate models. It is used 
   * to implement the class InterestRateModel. 
   * @see InterestRateModel
   */
  class IInterestRateModel
  {
  public:
    /** 
     * Virtual destructor. 
     */
    virtual ~IInterestRateModel(){};

    /** 
     * Virtual constructor. Constructs new implementation of the same model 
     * but with a different vector of event times.
     * 
     * @param rEventTimes New vector of event times for the model. 
     * The first element has to coincide with the initial time. 
     * @return Dynamically allocated implementation of the interface class IInterestRateModel with
     * the vector of event times \p rEventTimes. 
     */
    virtual IInterestRateModel *newModel(const std::vector<double> &rEventTimes) const = 0;

    /**
     * Constant reference to the underlying implementation of IModel. 
     * 
     * @return Constant reference to the underlying implementation of IModel. 
     */
    virtual const IModel &model() const = 0;

    /** 
     * Constructs discount factor with maturity \p dBondMaturity at event time with index \p iEventTime. 
     * @param iEventTime Index of event time where the discount factor is constructed. 
     * @param dBondMaturity The maturity of the discount factor. 
     * @return Discount factor with maturity \p dBondMaturity at event time with index \p iEventTime.
     */
    virtual Slice discount(unsigned iEventTime, double dBondMaturity) const = 0;
  };

  /** 
   * @brief  Concrete class for interest rate models. 
   * 
   * This is the main concrete class for interest rate models. 
   * @see IInterestRateModel
   */
  class InterestRateModel
  {
  public:
    /** 
     * The constructor. 
     * 
     * @param pNewModel Pointer to new implementation of IInterestRateModel. 
     */
    InterestRateModel(IInterestRateModel *pNewModel);

    /** 
     * Resets the vector of event times to \p rEventTimes and removes 
     * all path dependent state processes. After this operation, the member function \p model() 
     * will return a reference to a different "standard" implementation of IModel. 
     * 
     * @param rEventTimes The new vector of event times for the model. The front element of this 
     * vector should equal the initial time of the model. 
     */
    void assignEventTimes(const std::vector<double> &rEventTimes);

    /**
     * @copydoc IInterestRateModel::model
     */
    const IModel &model() const;

    /**
     * @copydoc IModel::eventTimes
     * The same as <code>model().eventTimes()</code>. 
     */
    const std::vector<double> &eventTimes() const;

    /**
     * Returns the initial time of the model. 
     * The same as <code>model().eventTimes().front()</code>
     * 
     * @return Initial time of the model. 
     */
    double initialTime() const;

    /** 
     * Constructs constant payoff in the amount \p dAmount at the event 
     * time with index \p iEventTime. 
     * 
     * @param iEventTime Index of event time. 
     * @param dAmount The amount of cash. 
     * @return Representation of constant payoff in the amount \p dAmount 
     * taking place at the event time with index \p iEventTime. 
     */
    Slice cash(unsigned iEventTime, double dAmount) const;

    /**
     * @copydoc IInterestRateModel::discount
     */
    Slice discount(unsigned iEventTime, double dBondMaturity) const;

  private:
    std::shared_ptr<IInterestRateModel> m_pModel;
  };
  /** @} */
} // namespace cfl

#include "cfl/Inline/iInterestRateModel.hpp"
#endif // of __cflInterestRateModel_hpp__
