#ifndef __cfl_Similar_hpp__
#define __cfl_Similar_hpp__

/**
 * @file Similar.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Implementation of models with identical state processes. 
 * @version 1.0
 * @date 2021-01-12
 * 
 * @copyright Copyright (c) 2020
 * 
 */

#include <cfl/Model.hpp>

namespace cfl
{
  /**
   * @ingroup cflCommonElements
   * 
   * @defgroup cflSimilar Similar financial models. 
   * 
   * This module facilitates implementations of financial models that
   * share the same state process (\a similar models). 
   * 
   * @see ISimilarRollback and Similar
   * @{
   */

  /**
   * @brief Interface class for the rollback operator of target model. 
   *
   * @see Similar
   */
  class ISimilarRollback
  {
  public:
    /**
     * Virtual destructor. 
     */
    virtual ~ISimilarRollback(){};
    
    /**
     * Runs rollback operator for the target model in the base model.
     * 
     * @param rSlice Payoff in the base model. 
     * @param iEventTime Index of event time. 
     */
    virtual void rollback(Slice &rSlice, unsigned iEventTime) const = 0;

    /**
     * Constant reference to the implementation of the base model. 
     */
    virtual const IModel &model() const = 0;
  };

  /**
   * @brief Constructs a similar target financial model from a base model. 
   * 
   * @see ISimilarRollback
   */
  class Similar
  {
  public:
    /**
     * Constructs a similar target financial model from the supplied 
     * implementation of its rollback operator in the base model. 
     * 
     * @param pNewSimilarRollback Pointer to new implementation of
     * ISimilarRollback.
     */
    Similar(ISimilarRollback *pNewSimilarRollback = 0);

    /**
     * @copydoc Similar::Similar
     */
    void assign(ISimilarRollback *pNewSimilarRollback);

    /**
     * Constant reference to the implementation of the target model. 
     */
    const IModel &model() const;

  private:
    std::shared_ptr<IModel> m_pModel;
  };
  /** @} */
} // namespace cfl

#include "cfl/Inline/iSimilar.hpp"
#endif // __cfl_Similar_hpp__
