#include <functional>
#include <algorithm>
#include <limits>
#include <cmath>
#include "cfl/Error.hpp"
#include "cfl/Data.hpp"

using namespace cfl;

namespace cflData
{
  std::function<bool(double)> belongs(const cfl::Function &rF, double dInitialTime)
  {
    return [rF, dInitialTime](double dT) {
      return rF.belongs(dT) && (dT >= dInitialTime);
    };
  }
} // namespace cflData

Function cfl::Data::discount(double dYield, double dInitialTime)
{
  std::function<double(double)> uDiscount = [dYield, dInitialTime](double dT) {
    PRECONDITION(dT >= dInitialTime);
    return std::exp(-dYield * (dT - dInitialTime));
  };
  return Function(uDiscount, dInitialTime);
}

Function cfl::Data::discount(const Function &rYield, double dInitialTime)
{
  std::function<double(double)> uDiscount = [rYield, dInitialTime](double dT) {
    PRECONDITION(dT >= dInitialTime);
    return std::exp(-rYield(dT) * (dT - dInitialTime));
  };
  return Function(uDiscount, cflData::belongs(rYield, dInitialTime));
}

//creation of volatility curves

Function cfl::Data::volatility(double dSigma, double dLambda, double dInitialTime)
{
  PRECONDITION(dSigma >= 0);
  std::function<double(double)> uVol = [dSigma, dLambda, dInitialTime](double dT) {
    PRECONDITION(dT >= dInitialTime);

    double dX = 2. * dLambda * (dT - dInitialTime);
    if (std::abs(dX) < cfl::EPS)
    {
      return dSigma;
    }
    return dSigma * std::sqrt((std::exp(dX) - 1.) / dX);
  };

  return Function(uVol, dInitialTime);
}

//creation of forward curves

Function cfl::Data::forward(double dSpot, double dCostOfCarry, double dInitialTime)
{
  std::function<double(double)> uForward = [dSpot, dCostOfCarry, dInitialTime](double dT) {
    PRECONDITION(dT >= dInitialTime);
    return dSpot * std::exp(dCostOfCarry * (dT - dInitialTime));
  };
  return Function(uForward, dInitialTime);
}

Function cfl::Data::forward(double dSpot, const Function &rCostOfCarry,
                            double dInitialTime)
{
  std::function<double(double)> uForward = [dSpot, rCostOfCarry, dInitialTime](double dT) {
    PRECONDITION(dT >= dInitialTime);
    return dSpot * std::exp(rCostOfCarry(dT) * (dT - dInitialTime));
  };
  return Function(uForward, cflData::belongs(rCostOfCarry, dInitialTime));
}

Function cfl::Data::forward(double dSpot, double dDividendYield,
                            const Function &rDiscount, double dInitialTime)
{
  std::function<double(double)> uForward = [dSpot, dDividendYield, rDiscount, dInitialTime](double dT) {
    PRECONDITION(dT >= dInitialTime);
    return dSpot * std::exp(-dDividendYield * (dT - dInitialTime)) / rDiscount(dT);
  };
  return Function(uForward, cflData::belongs(rDiscount, dInitialTime));
}

Function cfl::Data::assetShape(double dLambda, double dInitialTime)
{
  return discount(dLambda, dInitialTime);
}

Function cfl::Data::bondShape(double dLambda, double dInitialTime)
{
  std::function<double(double)> uShape = [dLambda, dInitialTime](double dTime) {
    PRECONDITION(dTime >= dInitialTime);
    double dT = dTime - dInitialTime;

    double dResult;
    if (std::abs(dLambda) <= cfl::EPS)
    {
      dResult = dT;
    }
    else
    {
      dResult = (1. - std::exp(-dLambda * dT)) / dLambda;
    }
    return dResult;
  };
  return Function(uShape, dInitialTime);
}

// CLASS: Swap

cfl::Data::Swap::Swap(const CashFlow &rCashFlow, bool bPayFloat)
    : CashFlow(rCashFlow), payFloat(bPayFloat) {}
