#include "cfl/Interp.hpp"
#include "cfl/Error.hpp"
#include <gsl/gsl_spline.h>

using namespace cfl;

// class Interp

cfl::Interp::Interp(IInterp *pNewP)
    : m_uP(pNewP) {}

// class GSL_Interp_Function

class GSL_Interp_Function : public IFunction
{
public:
  GSL_Interp_Function(const std::vector<double> &rArg,
                      const std::vector<double> &rVal,
                      const gsl_interp_type *pT)
      : m_uSpline(gsl_spline_alloc(pT, rArg.size()), &gsl_spline_free),
        m_uAcc(gsl_interp_accel_alloc(), &gsl_interp_accel_free)
  {
    PRECONDITION(rArg.size() == rVal.size());
    bool bSize = (rArg.size() >= gsl_interp_type_min_size(pT));
    PRECONDITION(bSize);
    if (!bSize)
    {
      throw(NError::size("interpolation: not enough points"));
    }
    gsl_spline_init(m_uSpline.get(), rArg.data(), rVal.data(), rArg.size());
    m_dL = rArg.front();
    m_dR = rArg.back();
    POSTCONDITION(m_dL < m_dR);
  }

  double operator()(double dX) const
  {
    bool bBelongs = belongs(dX);
    PRECONDITION(bBelongs);
    if (!bBelongs)
    {
      throw(NError::range("interpolation"));
    }
    return gsl_spline_eval(m_uSpline.get(), dX, m_uAcc.get());
  }

  bool belongs(double dX) const
  {
    return (dX >= m_dL) && (dX <= m_dR);
  }

private:
  std::unique_ptr<gsl_spline, decltype(&gsl_spline_free)> m_uSpline;
  std::unique_ptr<gsl_interp_accel, decltype(&gsl_interp_accel_free)> m_uAcc;
  double m_dL, m_dR;
};

//class GSL_Interp

class GSL_Interp : public IInterp
{
public:
  GSL_Interp(const gsl_interp_type *pT) : m_pT(pT) {}

  Function
  interpolate(const std::vector<double> &rArg, const std::vector<double> &rVal) const
  {
    //if size is 1 or 2 we do linear interpolation
    const gsl_interp_type * pT = (rArg.size() >2) ? m_pT : gsl_interp_linear;
    return Function(new GSL_Interp_Function(rArg, rVal, pT));
  }

private:
  const gsl_interp_type *m_pT;
};

// functions from NInterp

cfl::Interp cfl::NInterp::linear()
{
  return Interp(new GSL_Interp(gsl_interp_linear));
}

cfl::Interp cfl::NInterp::cspline()
{
  return Interp(new GSL_Interp(gsl_interp_cspline));
}

cfl::Interp cfl::NInterp::steffen()
{
  return Interp(new GSL_Interp(gsl_interp_steffen));
}

cfl::Interp cfl::NInterp::akima()
{
  return Interp(new GSL_Interp(gsl_interp_akima));
}

cfl::Interp cfl::NInterp::polynomial()
{
  return Interp(new GSL_Interp(gsl_interp_polynomial));
}
