#ifndef __test_all_Black_h__
#define __test_all_Black_h__

/**
 * @file Black.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Testing of Black models. 
 * @version 1.0
 * @date 2021-01-18
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "test/Print.hpp"
#include "cfl/BlackModel.hpp"

namespace test
{
  /**
   * @brief Testing in Black model. 
   * 
   */
  namespace Black
  {
    /** 
     *
     * @defgroup test_all_Black Testing in Black model. 
     * 
     * This module contains testing functions for 
     * the Black model on a single stock. 
     * 
     * @{
     */

    /**
     * The default interest rate. 
     * 
     */
    const double c_dYield = 0.07;

    /**
     * The default spot price. 
     * 
     */
    const double c_dSpot = 100;

    /**
     * The default dividend yield. 
     * 
     */
    const double c_dDividendYield = 0.02;

    /**
     * The default volatility of spot prices. 
     * 
     */
    const double c_dSigma = 0.2;

    /**
     * The default mean-reversion rate. 
     * 
     */
    const double c_dLambda = 0.05;

    /**
     * The default initial time. 
     * 
     */
    const double c_dInitialTime = 0;

    /**
     * Constructs data for Black model on a single stock. 
     * 
     * @param dYield The constant interest rate.
     * @param dSpot The spot price. 
     * @param dDividendYield The dividend yield. 
     * @param dSigma The spot volatility. 
     * @param dLambda The mean-reversion rate. 
     * @param dInitialTime The initial time. 
     * @return cfl::Black::Data 
     */
    cfl::Black::Data data(double dYield = c_dYield,
                          double dSpot = c_dSpot,
                          double dDividendYield = c_dDividendYield,
                          double dSigma = c_dSigma,
                          double dLambda = c_dLambda,
                          double dInitialTime = c_dInitialTime);

    /**
     * The default quality of model implementation. 
     * 
     */
    const double c_dQuality = 200;

    /**
     * Default implementation of Black model for testing purposes. 
     * 
     * @param dQuality The quality of model implementation. 
     * @return cfl::AssetModel 
     */
    cfl::AssetModel model(double dQuality = c_dQuality);

    /**
     * The default maturity. 
     * 
     */
    const double c_dMaturity = c_dInitialTime + 0.5;

    /**
     * The default notional. 
     * 
     */
    const double c_dNotional = 100.;

    /**
     * Returns some default vector of exercise times. 
     * 
     * @return std::vector<double> 
     */
    std::vector<double> exerciseTimes();

    /**
     * Returns some default vector of barrier times. 
     * 
     * @return std::vector<double> 
     */
    std::vector<double> barrierTimes();

    /**
     * The default interval for relative changes in the initial spot price. 
     * 
     */
    const double c_dInterval = 0.2;

    /**
     * The default number of values in the output. 
     * 
     */
    const unsigned c_iPoints = 10;

    /**
     * Prints results of valuation of an option on a single stock. 
     * 
     * @param f The testing function. 
     * @param rModel The reference to AssetModel. 
     */
    void report(cfl::MultiFunction (*f)(cfl::AssetModel &rModel),
                cfl::AssetModel &rModel);

    /**
     * Prints results of valuation of an option on a single stock. 
     * 
     * @param f The testing function; \p bPayFloat is a parameter for some
     * swap-like derivatives. 
     * @param rModel The reference to AssetModel. 
     */
    void report(cfl::MultiFunction (*f)(cfl::AssetModel &rModel, bool bPayFloat),
                cfl::AssetModel &rModel);
    /** @} */
  } // namespace Black
} // namespace test

#include "test/Inline/iBlack.hpp"
#endif // of __test_all_Black_h__
