#ifndef __Session2_hpp__
#define __Session2_hpp__

/**
 * @file Session2.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Session 2: interpolation and least-square fit
 * @version 0.1
 * @date 2021-02-12
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "cfl/Data.hpp"
#include "cfl/Interp.hpp"
#include "cfl/Fit.hpp"

/**
 * @mainpage Session 2: interpolation and least-square fit
 */

namespace prb
{
  /** 
   * @defgroup prbInterp Interpolation of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using interpolation.   
   * @{
   */
  
  /**
   * Computes a discount curve by interpolation of market yields.  The
   * construction of the discount curve is accomplished in 3 steps:
   * 1. We obtain the market yields from discount factors \p rDF.
   * 2. We get the market yield curve by interpolation with \p rInterp.
   * 3. We compute the discount curve from the yield curve.
   * 
   * @param rTimes The maturities of market discount factors.
   * @param rDF The market discount factors. 
   * @param dR The initial short-term interest rate. 
   * @param dInitialTime The initial time.
   * @param rInterp The interpolation method for yields.  
   *
   * @return cfl::Function The interpolated discount curve. 
   */
  cfl::Function discountYieldInterp(const std::vector<double> &rTimes,
				    const std::vector<double> &rDF,
				    double dR, double dInitialTime,
				    const cfl::Interp &rInterp);

  /**
   * Computes forward curve by applying the log-linear interpolation to 
   * the market spot and forward prices. 
   * 
   * @param dSpot The spot price of the stock. 
   * @param rDeliveryTimes The delivery times for market forward 
   * prices. 
   * @param rForwardPrices The  market forward prices.  
   * @param dInitialTime The initial time. 
   * 
   * @return The forward curve obtained from the market spot and
   * forward prices by log-linear interpolation.
   */
  cfl::Function 
  forwardLogLinInterp(double dSpot, 
		      const std::vector<double> & rDeliveryTimes, 
		      const std::vector<double> & rForwardPrices,
		      double dInitialTime); 
  /** @} */

  /** 
   * @defgroup prbFit Least-squares fitting of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using least-squares fitting.   
   * @{
   */
  
  /**
   * Returns forward curve in the form:
   * \f[
   * F(t) = S_0 \exp(q(t)(t-t_0)), \quad t\geq t_0, 
   * \f]
   * where cost-of-carry rate \f$q=q(t)\f$ is obtained by the
   * least-squares fit of the market cost-of-carry rates with method
   * \p rFit.
   * 
   * @param dSpot (\f$S_0\f$) The spot price of the stock. 
   * @param rDeliveryTimes The delivery times of forward contracts. 
   * @param rForwardPrices The market forward prices.  
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * @param rFit The fitting method. 
   * @param rErr The error function of the fit for forward prices. 
   * 
   * @return The fitted forward curve and its error function. We also
   * obtain the fitted coefficients and their covariance matrix
   * through \p rFit.
   */
  cfl::Function
  forwardCarryFit(double dSpot, const std::vector<double> &rDeliveryTimes,
                  const std::vector<double> &rForwardPrices, double dInitialTime,
                  cfl::Fit &rFit, cfl::Function &rErr);
  /** 
   * Returns the forward curve in the form:
   * \f[
   * F(t) = S_0 \exp(q(t)(t-t_0)), \quad t\geq t_0, 
   * \f]
   * where cost-of-carry rate \f$q=q(t)\f$ fits the market
   * cost-of-carry rates in the Black model for commodities.  We
   * recall that under the Black model, the log of spot price has the
   * form:
   * \f[
   * \log S_t = \log S_{t_0} + X_t, \quad t\geq t_0, 
   * \f]
   * where \f$X=(X_t)\f$ is an OU process: 
   * \f[
   *    dX_t =  (\theta - \lambda X_t) dt + \sigma dB_t,  
   *  \quad t\geq t_0, \quad X_0 = 0.  
   * \f]
   * Here \f$B=(B_t)\f$ is a Brownian motion and drift \f$\theta\f$
   * is the  fitting parameter.
   *
   * @param dSpot (\f$S_0\f$) The spot price of the stock. 
   * @param rTimes The maturities of forward contracts.
   * @param rForwards The market forward prices. 
   * @param dLambda (\f$\lambda\geq 0\f$) The mean-reversion rate. 
   * @param dSigma (\f$\sigma>0\f$) The volatility of \f$X_t = \log(S_t)\f$. 
   * @param dInitialTime (\f$t_0\f$) The initial time.
   * @param rErr The error function of the fit for forward prices. 
   * @param rParam The fitted coefficients, their covariance matrix, and
   * the total \f$\chi^2\f$ fitting error.
   * 
   * @return The fitted forward curve and its error function. We also
   * obtain the fitted coefficients and their covariance matrix.
   */
  cfl::Function
  forwardBlackFit(double dSpot, const std::vector<double> &rTimes,
                  const std::vector<double> &rForwards,
                  double dLambda, double dSigma, double dInitialTime,
                  cfl::Function &rErr, cfl::FitParam & rParam);
  /** @} */
}

#endif // of __Session2_hpp__
