#ifndef __SampleExam2_hpp__
#define __SampleExam2_hpp__

/**
 * @file SampleExam2.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief SampleExam2 for the course. 
 * @version 0.1
 * @date 2021-02-12
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "cfl/Data.hpp"
#include "cfl/Interp.hpp"
#include "cfl/Fit.hpp"
#include "cfl/AssetModel.hpp"
#include "cfl/InterestRateModel.hpp"

/**
 * @mainpage SampleExam2 for the course "Financial Computing with C++"
 */

namespace prb
{
  /** 
   * @defgroup prbInterp Interpolation of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using interpolation.   
   * @{
   */
  /** 
   * Computes implied volatility curve \f$\Sigma = \Sigma(t)\f$ by
   * applying linear interpolation to variance curve
   * \f[
   *      V(t) = \Sigma^2(t)(t - t_0), \quad t\geq t_0. 
   * \f]
   * The resulting volatility curve is constant on \f$[t_0,t_1]\f$. 
   * 
   * @param rTimes \f$(t_i)_{i=1,\dots,M}\f$ The maturities of the
   * market implied volatilities, \f$t_1>t_0\f$.
   * @param rVols The vector of market volatilities. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return The volatility curve obtained by linear interpolation
   * of market variances.
   */
  cfl::Function 
  volatilityVarLinInterp(const std::vector<double> & rTimes, 
			 const std::vector<double> & rVols, 
			 double dInitialTime);
  /** @} */  

  /** 
   * @defgroup prbFit Least-square fitting of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using least-square fitting.   
   * @{
   */
  /** 
   * Returns  discount curve
   * \f[
   *    d(t) = \exp(-\gamma(t)(t-t_0)), \quad t\geq t_0, 
   * \f]
   * by the least-squares fit of the stationary yield curve in the Hull
   * and White model
   * \f[
   * \gamma(t) = c_0 + c_1 \frac{1-e^{-\lambda (t-t_0)}}{\lambda(t-t_0)}, 
   * \quad t\geq t_0,   
   * \f]
   * to the market yields. Here, \f$c_0\f$ and \f$c_1\f$ are the fitting constants.
   *
   * @param rDiscountTimes The  maturities for market 
   * discount factors.
   * @param rDiscountFactors The  market discount factors. 
   * @param dLambda (\f$\lambda\geq 0\f$) The mean-reversion rate. 
   * @param dInitialTime (\f$t_0\f$) The initial time.
   * @param rErr The error function of the fit for discount factors. 
   * @param rParam The fitted coefficients, their covariance matrix, and
   * the total \f$\chi^2\f$ fitting error.
   * 
   * @return The fitted discount curve in Hull and White model and its
   * error function. We also obtain the fitted coefficients and their
   * covariance matrix.
   */
  cfl::Function discountHullWhiteFit(const std::vector<double> &rDiscountTimes,
                                     const std::vector<double> &rDiscountFactors,
                                     double dLambda, double dInitialTime,
                                     cfl::Function &rErr, cfl::FitParam & rParam);
  /** @} */

  /**
   * @defgroup cflAssetOptions Options on a single stock. 
   * 
   * This module deals with valuation of options on a single stock.
   * @{
   */

  /** 
   * Computes the value of the <strong>"BOOST" (Banking On Overall
   * Stability) option</strong>. This contract terminates at the first
   * barrier time when the stock price hits either of the barriers.
   * At the exit time the holder of the option receives the payoff
   * given by the product of notional amount \p dNotional on the
   * percentage of the barrier times the stock price spends between
   * the barriers.  If the price of the stock never leaves the
   * barriers, then at the last barrier time the holder receives
   * notional amount \p dNotional.
   * 
   * @param dNotional The notional amount.  
   * @param dLowerBarrier The lower barrier.  
   * @param dUpperBarrier The upper barrier.  
   * @param rBarrierTimes The vector of barrier times. The first
   * time is greater than the initial time. 
   * @param rModel The reference to an implementation of cfl::AssetModel. 
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.  
   */
  cfl::MultiFunction
  boost(double dNotional, double dLowerBarrier,
	double dUpperBarrier,
	const std::vector<double> &rBarrierTimes,
	cfl::AssetModel &rModel);  
  /** @} */

  /**
   * @defgroup cflInterestRateOptions Options on interest rates. 
   * 
   * This module deals with valuation of standard and barrier 
   * options on interest rates.
   * 
   * @{
   */

  /** 
   * Computes the price of <strong>puttable bond with resettable
   * coupon</strong> at its issue time. In this contract, after a
   * coupon payment the issuer can reset the coupon rate from original
   * (higher) value <code>rBond.rate</code> to (lower) reset value \p
   * dResetCouponRate. After that, at any payment time greater or
   * equal the reset time and less than the maturity, the holder can
   * sell the bond back to the issuer for redemption value
   * <code>dRedemptionPrice*rBond.notional</code>.
   * 
   * @param rBond The parameters of the underlying coupon bond. The
   * initial coupon rate is given by <code>rBond.rate</code>. 
   * @param dResetCouponRate The coupon rate after reset. Usually,
   * this rate is smaller than the original coupon rate. 
   * @param dRedemptionPrice The redemption price of the bond as the percentage
   * of the notional.  Typically, <code>dRedemptionPrice<1</code>. 
   * @param rModel The reference to the implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction resetCouponPutBond(const cfl::Data::CashFlow &rBond,
                                        double dResetCouponRate, double dRedemptionPrice,
                                        cfl::InterestRateModel &rModel);  
  /** @} */  
}

#endif // of __SampleExam2_hpp__
