#ifndef __cflHullWhiteModel_hpp__
#define __cflHullWhiteModel_hpp__

/**
 * @file HullWhiteModel.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Implementation of Hull and White model. 
 * @version 1.0
 * @date 2021-01-12
 * 
 * @copyright Copyright (c) 2020
 * 
 */

#include "cfl/Data.hpp"
#include "cfl/Brownian.hpp"
#include "cfl/InterestRateModel.hpp"

namespace cfl
{
  /** 
   * @ingroup cflInterestRateModel
   *
   * @defgroup cflHullWhite Hull and White model for interest rates.
   * 
   * This module contains implementation of the classical 
   * Hull and White model for interest rates. 
   * 
   * @see cfl::InterestRateModel and cfl::Brownian.
   * @{
   */

  /** 
   * @brief  Hull and White model for interest rates. 
   * 
   * This namespace contains classes and functions related with 
   * Hull and White model for interest rates. 
   */
  namespace HullWhite
  {

    /** 
     * @brief  The parameters of Hull and White model. 
     * 
     * This class defines the parameters of the Hull and White model. The set of parameters 
     * consists of discount, shape, and volatility curves. 
     */
    class Data
    {
    public:
      /**
       * Default constructor. 
       */
      Data(){};

      /** 
       * Constructs parameters of Hull and White model.
       * 
       * @param rDiscount A discount curve.
       * @param rVolatility A volatility curve for the factor process. 
       * @param rShape A shape function. This function defines the shape of 
       * changes in discount or yield curves. 
       * @param dInitialTime Initial time given as year fraction.
       */
      Data(const Function &rDiscount,
           const Function &rVolatility,
           const Function &rShape,
           double dInitialTime);

      /** 
       * Constructs parameters of Hull and White model with 
       * stationary volatility curve.
       * 
       * @param rDiscount A discount curve.
       * @param dSigma The volatility of short-term interest rate. 
       * @param dLambda The mean-reversion coefficient of short-term 
       * interest rate under the money market martingale measure. 
       * @param dInitialTime Initial time given as year fraction.
       */
      Data(const Function &rDiscount,
           double dSigma,
           double dLambda,
           double dInitialTime);

      /**
     * @copydoc InterestRateModel::initialTime
     */
      double initialTime() const;

      /** 
       * Accessor function to discount curve. 
       * 
       * @return The discount curve. 
       */
      const Function &discount() const;

      /** 
       * Accessor function to volatility curve. 
       * 
       * @return The volatility curve for the state process. 
       */
      const Function &volatility() const;

      /** 
       * Accessor function to "shape" curve.  
       * 
       * @return The shape function. This function defines the shape of 
       * changes in discount or yield curves. 
       */
      const Function &shape() const;

    private:
      Function m_uDiscount;
      Function m_uVol;
      Function m_uShape;
      double m_dInitialTime;
    };

    /** 
     * Implements \p InterestRateModel as Hull and White model. 
     * 
     * @param rData The parameters of Hull and White model. 
     * @param dInterval The interval of initial values for short-term 
     * interest rate. 
     * @param dQuality The trade-off between speed and accuracy. 
     * @return \p InterestRateModel as Hull and White model. 
     */
    InterestRateModel model(const Data &rData, double dInterval, double dQuality);

    /** 
     * Implements \p InterestRateModel as Hull and White model.
     * 
     * @param rData The parameters of Hull and White model. 
     * @param dInterval The interval of initial values for short-term 
     * interest rate. 
     * @param rBrownian Implementation of class Brownian. It is used to 
     * price standard derivatives. 
     * @return \p InterestRateModel as Hull and White model. 
     */
    InterestRateModel model(const Data &rData, double dInterval, const Brownian &rBrownian);
  } // namespace HullWhite
  /** @} */
} // namespace cfl

#include "cfl/Inline/iHullWhiteModel.hpp"
#endif // of __cflHullWhiteModel_hpp__
