#include <random>
#include "cfl/Macros.hpp"
#include "test/Output.hpp"
#include "test/Print.hpp"

using namespace std;
using namespace cfl;

using namespace std;
using namespace cfl;

// accessor functions

std::valarray<double> test::getArg(double dL, double dR, unsigned iN)
{
  ASSERT(iN > 0);
  std::valarray<double> uResult(iN);
  double dH = (dR - dL) / (iN - 1);
  double dX = dL;
  for (unsigned iI = 0; iI < iN; iI++)
  {
    uResult[iI] = dX;
    dX += dH;
  }
  uResult[uResult.size() - 1] = dR;
  return uResult;
}

std::vector<double> test::getTimes(double dInitialTime, double dMaturity, unsigned iN)
{
  std::valarray<double> uArg = getArg(dInitialTime, dMaturity, iN + 1);
  std::vector<double> uTimes(std::begin(uArg) + 1, std::end(uArg));
  return uTimes;
}

std::valarray<double> test::getRandArg(double dL, double dR, unsigned iN)
{
  ASSERT(iN > 0);
  std::valarray<double> uResult(iN);
  std::minstd_rand uGen(1);
  std::uniform_real_distribution<double> uRand(dL, dR);
  for (unsigned iI = 0; iI < iN; iI++)
  {
    uResult[iI] = uRand(uGen);
  }
  std::sort(begin(uResult), end(uResult));
  POSTCONDITION((dL < uResult[0]) && (uResult[uResult.size() - 1] < dR));
  return uResult;
}

std::valarray<double> test::
    getValues(const Function &rF, const std::valarray<double> &rArg)
{
  std::valarray<double> uResult(rArg.size());
  std::transform(begin(rArg), end(rArg), begin(uResult), [&rF](double dX) {
    return rF(dX);
  });
  return uResult;
}

// print functions

void test::compare(const std::valarray<double> &rExact,
                   const std::valarray<double> &rNum,
                   const std::string &rTitle, unsigned iColumn,
                   unsigned iSpace, unsigned iMaxRows)
{
  PRECONDITION(rExact.size() == rNum.size());
  unsigned iColumns = 3;

  std::vector<std::valarray<double>>
      uResults(iColumns, std::valarray<double>(rExact.size()));
  uResults[0] = rExact;
  uResults[1] = rNum;
  uResults[2] = std::abs(uResults[0] - uResults[1]);

  std::vector<std::string> uHeads(iColumns);
  uHeads[0] = "exact";
  uHeads[1] = "numeric";
  uHeads[2] = "error";

  printTable(uResults, uHeads, rTitle, iColumn, iSpace, iMaxRows);
}

void test::print(double dValue, const std::string &sMessage, bool bExtraLine)
{
  std::string sM(sMessage);
  sM += std::string(" = ");
  std::cout << sM.c_str() << dValue << endl;
  if (bExtraLine)
  {
    std::cout << endl;
  }
}

void test::print(const std::string &sMessage)
{
  std::cout << sMessage.c_str() << endl
            << endl;
}

void test::printValues(const cfl::Function &rF, const std::valarray<double> &rArg, const std::string &rTitle)
{
  std::valarray<double> uValues = getValues(rF, rArg);
  print(begin(uValues), end(uValues), rTitle);
}

void test::printTable(const std::vector<std::valarray<double>> &rValues,
                      const std::vector<std::string> &rNames,
                      const std::string &sMessage, unsigned iColumn,
                      unsigned iSpace, unsigned iMaxRows)
{
  PRECONDITION(rValues.size() == rNames.size());
  cout << sMessage.c_str() << endl
       << endl;

  for (unsigned i = 0; i < rValues.size(); i++)
  {
    std::cout << std::setw(iColumn) << rNames[i].c_str() << std::setw(iSpace) << "";
  }
  std::cout << endl;

  unsigned iSize = rValues.front().size();
  unsigned iRows = std::min(iSize, iMaxRows);
  unsigned iStart = (iSize - iRows) / 2;
  unsigned iEnd = (iSize + iRows) / 2;
  iEnd = min(iEnd, iSize);

  for (unsigned j = iStart; j < iEnd; j++)
  {
    for (unsigned i = 0; i < rValues.size(); i++)
    {
      ASSERT(rValues[i].size() == iSize);
      std::cout << std::setw(iColumn) << rValues[i][j] << std::setw(iSpace) << "";
    }
    std::cout << endl;
  }
  std::cout << std::endl;
}

void test::printTable(const std::vector<std::vector<double>> &rValues,
                      const std::vector<std::string> &rNames,
                      const std::string &sMessage, unsigned iColumn,
                      unsigned iSpace, unsigned iMaxRows)
{
  unsigned iSize = rValues.front().size();
  std::vector<std::valarray<double>> uV(rValues.size(), std::valarray<double>(iSize));
  for (unsigned i = 0; i < rValues.size(); i++)
  {
    std::copy(rValues[i].begin(), rValues[i].end(), begin(uV[i]));
  }
  printTable(uV, rNames, sMessage, iColumn, iSpace, iMaxRows);
}

void test::printTable(const std::vector<cfl::Function> &rF, const std::vector<std::string> &rNames,
                      const std::valarray<double> &rArg, const std::string &sMessage,
                      unsigned iColumn, unsigned iSpace, const std::string &sArg)
{
  PRECONDITION(rF.size() == rNames.size());
  std::vector<std::string> uNames(rNames.size() + 1);
  uNames.front() = sArg;
  std::copy(rNames.begin(), rNames.end(), uNames.begin() + 1);
  std::vector<std::valarray<double>>
      uValues(rF.size() + 1, std::valarray<double>(rArg.size()));
  uValues.front() = rArg;
  for (unsigned i = 0; i < rF.size(); i++)
  {
    uValues[i + 1] = getValues(rF[i], rArg);
  }
  printTable(uValues, uNames, sMessage, iColumn, iSpace, uValues.front().size());
}

double total_err(const cfl::Function &rErr, const std::valarray<double> &rArg)
{
  std::valarray<double> uErr = test::getValues(rErr, rArg);
  double dErr = std::inner_product(begin(uErr), end(uErr), begin(uErr), 0.);
  return std::sqrt(dErr);
}

void test::printTotalErr(const cfl::Function &rEstErr, const cfl::Function &rActErr,
                         const std::valarray<double> &rArg)
{
  cout << "Total estimated error = " << total_err(rEstErr, rArg)
       << ". Total actual error = " << total_err(rActErr, rArg) << "." << endl
       << endl;
}

//temp from old version

void test::printRisk(const cfl::Function &rOption, double dShift)
{
  cout << "RISK REPORT: " << endl
       << endl;
  double dCenter = 0.;
  double dL = -dShift;
  double dR = dShift;
  double dPrice = rOption(dCenter);
  cout << "price = " << dPrice << endl;
  if (rOption.belongs(dR) && rOption.belongs(dL))
  {
    double dValueLeft = rOption(dL);
    double dValueRight = rOption(dR);
    cout << "delta = " << (dValueRight - dValueLeft) / (2. * dShift) << endl;
    cout << "one percent gamma = " << 0.01 * (dValueRight - 2. * dPrice + dValueLeft) / (dShift * dShift) << endl
         << endl;
  }
}

namespace testPrint
{
  void print(const cfl::Data::CashFlow &rCashFlow, const std::string &rName)
  {
    std::string sM(rName);
    sM += std::string(":");
    cout << sM.c_str() << endl;
    cout << "notional = " << rCashFlow.notional << endl;
    cout << "period between payments = " << rCashFlow.period << endl;
    cout << "number of payments = " << rCashFlow.numberOfPayments << endl;
    cout << "rate = " << rCashFlow.rate << endl;
  }
} // namespace testPrint

void test::printCashFlow(const cfl::Data::CashFlow &rCashFlow, const std::string &rName)
{
  testPrint::print(rCashFlow, rName);
  cout << endl;
}

void test::printSwap(const cfl::Data::Swap &rSwap, const std::string &rName)
{
  testPrint::print(cfl::Data::CashFlow(rSwap), rName);
  if (rSwap.payFloat)
  {
    cout << "we pay float and receive fixed" << endl;
  }
  else
  {
    cout << "we pay fixed and receive float" << endl;
  }
  cout << endl;
}
