#ifndef __SampleExam1_hpp__
#define __SampleExam1_hpp__

/**
 * @file SampleExam1.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief SampleExam1 for the course. 
 * @version 0.1
 * @date 2021-02-12
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "cfl/Data.hpp"
#include "cfl/Interp.hpp"
#include "cfl/AssetModel.hpp"
#include "cfl/InterestRateModel.hpp"

/**
 * @mainpage SampleExam1 for the course "Financial Computing with C++"
 */

namespace prb
{
  /** 
   * @defgroup prbData Data structures. 
   *
   * This module constructs input data curves for financial models. 
   * @{
   */

  /** 
   * Computes yield curve \f$\gamma=(\gamma(t))_{t\geq t_0}\f$ for the Vasicek
   * model, where short-term interest rate \f$r=(r_t)\f$ is an OU
   * (Ornstein-Uhlenbeck) process driven by Brownian motion \f$B=(B_t)\f$: 
   * \f[
   *   dr_t=  (\theta - \lambda r_t) dt + \sigma dB_t,  
   *  \quad t\geq t_0.  
   * \f]
   * 
   * @param dTheta (\f$\theta\f$) The drift. 
   * @param dLambda (\f$\lambda\geq 0\f$) The mean-reversion rate. 
   * @param dSigma (\f$\sigma\geq 0\f$) The volatility. 
   * @param dR0 (\f$r_0\f$) The initial short-term interest rate. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return Yield curve for the Vasicek model of interest rates.
   */
  cfl::Function yieldVasicek(double dTheta, double dLambda, double dSigma,
                             double dR0, double dInitialTime);

  /** @} */  

  /** 
   * @defgroup prbInterp Interpolation of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using interpolation.   
   * @{
   */
  
  /**
   * Computes the forward exchange rate curve by the log-linear
   * interpolation of the market spot and forward fx rates.
   * 
   * @param dSpotFX The spot exchange rate. 
   * @param rDiscountTimes The maturities of discount factors. 
   * @param rDomesticDiscountFactors The domestic discount factors.
   * @param rForeignDiscountFactors The foreign discount factors.
   * @param dInitialTime The initial time.
   * 
   * @return The forward exchange rate curve obtained by the
   * log-linear interpolation of the market forward rates.
   */
  cfl::Function 
  forwardFXLogLinInterp(double dSpotFX, 
			const std::vector<double> & rDiscountTimes, 
			const std::vector<double> & rDomesticDiscountFactors,
			const std::vector<double> & rForeignDiscountFactors,
			double dInitialTime);  
  /** @} */
  
  /**
   * @defgroup cflAssetOptions Options on a single stock. 
   * 
   * This module deals with valuation of options on a single stock.
   * @{
   */

  /** 
   * Computes the value of the <strong>up-range-out put option</strong>. 
   * The option is canceled after first \p iOutTimes barrier times, 
   * when the stock price is above \p dUpperBarrier. Otherwise, 
   * it behaves as standard European put option with maturity \p dMaturity
   * and strike \p dStrike. 
   *
   * @param dUpperBarrier The upper barrier.  
   * @param iOutTimes The number of barrier events ("stock price is 
   * above \p dUpperBarrier") that cancels the option. 
   * @param rBarrierTimes The vector of barrier times. The first
   * time is greater than the initial time. The last barrier time is less 
   * than \p dMaturity. 
   * @param dStrike The strike of the put option. 
   * @param dMaturity The maturity of the option.  
   * @param rModel The reference to an implementation of cfl::AssetModel. 
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.  
   */
  cfl::MultiFunction upRangeOutPut(double dUpperBarrier, unsigned iOutTimes,
				   const std::vector<double> &rBarrierTimes,
				   double dStrike, double dMaturity,
				   cfl::AssetModel &rModel);  
  /** @} */

  /**
   * @defgroup cflInterestRateOptions Options on interest rates. 
   * 
   * This module deals with valuation of standard and barrier 
   * options on interest rates.
   * 
   * @{
   */

  /** 
   * Computes the <strong>futures price of  the cheapest bond to
   * deliver</strong>. 
   * The set of times, when the futures price is evaluated is given by 
   * \f[
   * t_i = t_0 + i\frac{T-t_0}{M}, \quad i=1,\dots,M, 
   * \f]
   * where \f$t_0=\f$ \p rModel.initialTime(), \f$T=\f$ \p
   * dFuturesMaturity, and \f$M=\f$ \p iFuturesTimes.  At the
   * maturity, the seller of the contract can deliver any bond from \p
   * rBonds.
   * 
   * @param dFuturesMaturity The maturity of the futures contract. 
   * @param iFuturesTimes The number of futures times, that is, the
   * times when the futures price is determined. 
   * @param rBonds The parameters of the underlying bonds. We assume
   * that the issue times for all bonds equal \p dFuturesMaturity. 
   * @param rModel The reference to the implementation of
   * cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction
  futuresOnCheapToDeliver(double dFuturesMaturity,
                          unsigned iFuturesTimes,
                          const std::vector<cfl::Data::CashFlow> &rBonds,
                          cfl::InterestRateModel &rModel);  
  /** @} */  
}

#endif // of __SampleExam1_hpp__
