#ifndef __Session1_hpp__
#define __Session1_hpp__

/**
 * @file Session1.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Session 1: data curves for financial models
 * @version 0.1
 * @date 2021-02-12
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "cfl/Data.hpp"

/**
 * @mainpage Session 1: data curves for financial models
 */

namespace prb
{
  /** 
   * @defgroup prbData Data structures. 
   *
   * This module constructs input data curves for financial models. 
   * @{
   */

  /** 
   * Computes the continuously compounded yield curve from a discount
   * curve. We recall that 
   * \f[
   *  D(t) = e^{-\gamma(t)(t-t_0)}, \quad t\geq t_0. 
   * \f]
   * 
   * @param rDiscount \f$(D(t))_{t\geq t_0}\f$ The discount curve. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return \f$(\gamma(t))_{t\geq t_0}:\f$ the continuously compounded
   * yield curve.
   */
  cfl::Function yield(const cfl::Function &rDiscount, double dInitialTime);

  /** 
   * Returns the stationary implied volatility curve for the Black
   * model:
   * \f[ 
   * \Sigma(t) = \sigma \sqrt{\frac{1 -\exp(-2\lambda
   *   (t-t_0))}{2\lambda (t-t_0)}}, \quad t\geq t_0.
   * \f]
   * 
   * @param dSigma (\f$\sigma\geq 0\f$) The short-term volatility. 
   * @param dLambda (\f$\lambda\geq 0\f$) The mean-reversion rate. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return The stationary implied volatility curve in Black model.
   */
  cfl::Function
  volatilityBlack(double dSigma, double dLambda, double dInitialTime);

  /** 
   * Computes the Nelson-Siegel yield curve:
   * \f[
   *  \gamma(t) = c_0 + c_1 \frac{1-e^{-\lambda(t-t_0)}}{\lambda (t-t_0)} +
   *  c_2 \left(\frac{1-e^{-\lambda(t-t_0)}}{\lambda (t-t_0)}  -
   *     e^{-\lambda (t-t_0)}\right), \quad t\geq t_0. 
   * \f]
   *
   * @param dC0 (\f$c_0\f$) The first constant
   * @param dC1 (\f$c_1\f$) The second constant. 
   * @param dC2 (\f$c_2\f$) The third constant. 
   * @param dLambda (\f$\lambda>0\f$) The mean-reversion rate. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return The yield curve for the Nelson-Siegel model. 
   */
  cfl::Function
  yieldNelsonSiegel(double dC0, double dC1, double dC2,
                    double dLambda, double dInitialTime);

  /** 
   * Computes the curve of forward prices for a dividend paying
   * stock. The buyer pays forward price \f$F(t)\f$ at delivery time \f$t\f$
   * and then receives the stock. If \f$t\f$ is a dividend time, then the
   * buyer gets the dividend paid at \f$t\f$.
   *
   * @param dSpot The spot price of the stock. 
   * @param rDividendsTimes The dividend times.
   * @param rDividends The dividend payments. 
   * @param rDiscount The discount curve. 
   * @param dInitialTime The initial time.
   * 
   * @return The forward price curve for the stock. 
   */
  cfl::Function forwardStockDividends(double dSpot,
                                      std::vector<double> &rDividendsTimes,
                                      std::vector<double> &rDividends,
                                      const cfl::Function &rDiscount,
                                      double dInitialTime);
  /** @} */
} // namespace prb

#endif // of __Session1_hpp__
