#ifndef __SampleExam3_hpp__
#define __SampleExam3_hpp__

/**
 * @file SampleExam3.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief SampleExam3 for the course. 
 * @version 0.1
 * @date 2021-02-12
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "cfl/Data.hpp"
#include "cfl/Interp.hpp"
#include "cfl/Fit.hpp"
#include "cfl/AssetModel.hpp"
#include "cfl/InterestRateModel.hpp"

/**
 * @mainpage SampleExam3 for the course "Financial Computing with C++"
 */

namespace prb
{
  /** 
   * @defgroup prbInterp Interpolation of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using interpolation.   
   * @{
   */
  
  /** 
   * Computes the discount curve from a family of swap rates.  The
   * computation is done in two steps:
   * 1. We deduce discount factors for the maturities of swap contracts.
   * 2. We apply log-linear interpolation to these discount factors.
   * 
   * @param rSwapRates \f$(R_i)_{i=1,\dots,M}\f$ The market swap
   * rates. The maturities of the contracts have the form:
   * \f[
   * t_i = t_0 + i\delta t, \quad i=1,\dots,M.
   * \f] 
   * @param dPeriod (\f$\delta t\f$) The time interval between two
   * payments in a swap.
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return The discount curve obtained from the market swap rates by
   * log-linear interpolation of discount factors.
   */
  cfl::Function 
  discountSwapLogLinInterp(const std::vector<double> & rSwapRates, 
			   double dPeriod, double dInitialTime);
  /** @} */  

  /** 
   * @defgroup prbFit Least-square fitting of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using least-square fitting.   
   * @{
   */
  
  /**
   * Returns the forward exchange curve in the form:
   * \f[
   * F(t) = S_0 \exp(q(t)(t-t_0)), \quad t\geq t_0, 
   * \f]
   * where cost-of-carry rate \f$q=q(t)\f$ is obtained by the
   * least-squares fit of the market cost-of-carry rates to the
   * Svensson curve:
   * \f[
   *  q(t) = c_0 + c_1 \frac{1-e^{-\lambda_1(t-t_0)}}{\lambda_1 (t-t_0)} +
   *  c_2 \left(\frac{1-e^{-\lambda_1(t-t_0)}}{\lambda_1 (t-t_0)}  -
   *     e^{-\lambda_1 (t-t_0)}\right) +
   *  c_3 \left(\frac{1-e^{-\lambda_2(t-t_0)}}{\lambda_2 (t-t_0)}  -
   *     e^{-\lambda_2 (t-t_0)}\right) , \quad t\geq t_0.  
   * \f]
   * Here \f$c_0\f$, \f$c_1\f$, \f$c_2\f$, and \f$c_3\f$ are the
   * fitting constants.
   * 
   * @param dSpotFX The spot exchange rate. 
   * @param rTimes The  maturities for discount factors. 
   * @param rDomesticDiscountFactors The  domestic discount factors. 
   * @param rForeignDiscountFactors The  foreign discount factors.
   * @param dLambda1 (\f$\lambda_1\f$) The first mean-reversion rate. 
   * @param dLambda2 (\f$\lambda_2\f$) The second mean-reversion rate. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * @param rErr The error function of the fit for forward prices. 
   * @param rParam The fitted coefficients, their covariance matrix, and
   * the total \f$\chi^2\f$ fitting error.
   * 
   * @return The fitted forward curve and its error function. We also
   * obtain the fitted coefficients and their covariance matrix.
   */
  cfl::Function
  forwardFXSvenssonFit(double dSpotFX, const std::vector<double> &rTimes,
                       const std::vector<double> &rDomesticDiscountFactors,
                       const std::vector<double> &rForeignDiscountFactors,
                       double dLambda1, double dLambda2,
                       double dInitialTime, cfl::Function &rErr,
		       cfl::FitParam & rParam);
  /** @} */

  /**
   * @defgroup cflAssetOptions Options on a single stock. 
   * 
   * This module deals with valuation of options on a single stock.
   * @{
   */

  /** 
   * Computes strike \f$K\f$ of <strong>asset variance swap</strong>.
   * The square of the strike is the forward price paid by the buyer
   * at maturity \p dMaturity in return for the annualized variance of
   * the stock price. In other words, the payoff of the long position
   * in the forward contract at the maturity is given by
   * \f[
   * V(T) = \frac1{T - t_0} \sum_{i=1}^N 
   * \left(\ln\left(\frac{S(t_i)}{S(t_{i-1})} \right)\right)^2 - K^2.
   * \f]
   * Here \f$S(t_i)\f$ is the spot price at  time 
   * \f[
   *   t_i = t_0 + i \frac{T - t_0}{N}, \quad 1\leq i\leq N, 
   * \f]
   * \f$T\f$ is the maturity (\p dMaturity), \f$t_0\f$ is the initial
   * time (\p rModel.initialTime()) and \f$N\f$ is the number of times
   * used in the computation of the variance (\p iNumberOfTimes)
   * 
   * @param dMaturity The maturity of the variance swap. 
   * @param iNumberOfTimes The number of times participating in the 
   * computation of the variance
   * @param rModel The reference to an implementation of cfl::AssetModel. 
   * 
   * @return The strike of the asset variance swap as the function of the 
   * initial values of the state processes in the model. 
   */
  cfl::MultiFunction varianceSwapStrike(double dMaturity, unsigned iNumberOfTimes,
					cfl::AssetModel &rModel);
  /** @} */

  /**
   * @defgroup cflInterestRateOptions Options on interest rates. 
   * 
   * This module deals with valuation of standard and barrier 
   * options on interest rates.
   * 
   * @{
   */
   /** 
   * Computes the price of <strong>callable capped floater</strong>. 
   * In this contract, at an exercise time \f$t_i\f$ the holder of the
   * option receives the coupon 
   * \f[
   *    N \delta t \times \min(L(t_{i-1},t_i)+ \delta L, C)
   * \f]
   * where \f$L(s,t)\f$ is the LIBOR rate computed at \f$s\f$ for maturity 
   * \f$t\f$, \f$N\f$  is the notional amount (\p rCap.notional), 
   * \f$\delta L\f$ is the spread over LIBOR (\p dLiborSpread), 
   * \f$C\f$ is the cap rate (\p rCap.rate) and 
   * \f$\delta t = t_{i} - t_{i-1}\f$ 
   * is the time interval between two payments (\p rCap.period). 
   * The seller can terminate the contract at any exercise time. 
   * In this case, in addition to the above coupon he pays the
   * notional. If the contract has not been terminated before, 
   * then at maturity the holder receives the above coupon plus notional.
   * 
   * @param rCap The parameters of the contract. Here <code>rCap.rate</code>
   *  defines the cap rate. 
   * @param dLiborSpread The spread over LIBOR rate. 
   * @param rModel The reference to the implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction callableCappedFloater(const cfl::Data::CashFlow &rCap,
                                           double dLiborSpread,
                                           cfl::InterestRateModel &rModel);  
  /** @} */  
}

#endif // of __SampleExam3_hpp__
