#ifndef __Homework2_hpp__
#define __Homework2_hpp__

/**
 * @file Homework2.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Homework 2: interpolation and least-square fit
 * @version 0.1
 * @date 2021-02-12
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "cfl/Data.hpp"
#include "cfl/Interp.hpp"
#include "cfl/Fit.hpp"

/**
 * @mainpage Homework 2: interpolation and least-square fit
 */

namespace prb
{
  /** 
   * @defgroup prbInterp Interpolation of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using interpolation.   
   * @{
   */

  /**
   * Computes forward curve
   * \f[
   * F(t) = S_0 e^{q(t)(t-t_0)}, \quad t\in [t_0,t_M], 
   * \f]
   * where cost-of-carry rate \f$q=q(t)\f$ is obtained by the linear
   * interpolation to the market cost-of-carry rates. On the first
   * interval \f$[t_0,t_1]\f$, cost-of-carry function \f$q=q(t)\f$ is
   * assumed to be constant.
   * 
   * @param dSpotFX (\f$S_0\f$) The spot exchange rate. 
   * @param rDiscountTimes \f$(t_i)_{i=1,\dots,M}\f$ The maturities of
   * discount factors.
   * @param rDomesticDiscountFactors The domestic discount factors.
   * @param rForeignDiscountFactors The foreign discount factors.
   * @param dInitialTime (\f$t_0\f$) The initial time.
   * 
   * @return The forward exchange rate curve obtained by the linear
   * interpolation of the market cost-of-carry rates.
   */
  cfl::Function 
  forwardFXCarryLinInterp(double dSpotFX, 
			  const std::vector<double> & rDiscountTimes, 
			  const std::vector<double> & rDomesticDiscountFactors,
			  const std::vector<double> & rForeignDiscountFactors,
			  double dInitialTime);

  /**
   * Computes the discount curve by interpolation of logs of market discount factors. 
   * The construction of the discount curve is accomplished in 3 steps: 
   * 1. We compute the logs of  discount factors \p rDF.
   * 2. We interpolate the logs of the discount factors with \p rInterp. 
   * 3. We recover the discount curve from its logarithmic interpolation. 
   * 
   * @param rTime The maturities of market discount factors.
   * @param rDF The market discount factors. 
   * @param dInitialTime The initial time.
   * @param rInterp The interpolation method for logs of discount factors.   
   *
   * @return cfl::Function The interpolated discount curve. 
   */
  cfl::Function
  discountLogInterp(const std::vector<double> &rTime, const std::vector<double> &rDF,
		    double dInitialTime, const cfl::Interp &rInterp);
  /** @} */  

  /** 
   * @defgroup prbFit Least-square fitting of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using least-square fitting.   
   * @{
   */
  /** 
   * Returns the stationary implied volatility curve for the Black model by
   * the least-squares approximation of the market implied volatilities.
   * The stationary volatility form for Black model is given by
   * \f[ 
   * \Sigma(t) = \sigma \sqrt{\frac{1 -\exp(-2\lambda
   *   (t-t_0))}{2\lambda (t-t_0)}}, \quad t\geq t_0, 
   * \f]
   * where short-term volatility \f$ \sigma \f$ is the  fitted
   * parameter.
   * 
   * @param rTimes The maturities of the market implied volatilities.
   * @param rVols The market implied volatilities. 
   * @param dLambda (\f$\lambda\geq 0\f$) The mean-reversion rate. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * @param rErr The error function of the fit for implied volatilities.
   * @param rParam The fitted coefficients, their covariance matrix, and
   * the total \f$\chi^2\f$ fitting error.
   *
   * @return The fitted volatility curve and its error function. We also
   * obtain the fitted coefficients and their covariance matrix.
   */
  cfl::Function
  volatilityBlackFit(const std::vector<double> &rTimes,
                     const std::vector<double> &rVols,
                     double dLambda, double dInitialTime,
                     cfl::Function &rErr, cfl::FitParam & rParam);

  /**
   * Returns discount curve
   * \f[
   *    d(t) = \exp(-\gamma(t)(t-t_0)), \quad t\geq t_0, 
   * \f]
   * by the least-squares fit of Svensson yield curve
   * \f[
   *  \gamma(t) = c_0 + c_1 \frac{1-e^{-\lambda_1(t-t_0)}}{\lambda_1 (t-t_0)} +
   *  c_2 \left(\frac{1-e^{-\lambda_1(t-t_0)}}{\lambda_1 (t-t_0)}  -
   *     e^{-\lambda_1 (t-t_0)}\right) +
   *  c_3 \left(\frac{1-e^{-\lambda_2(t-t_0)}}{\lambda_2 (t-t_0)}  -
   *     e^{-\lambda_2 (t-t_0)}\right) , \quad t\geq t_0,  
   * \f]
   * to continuously compounded market yields. Here \f$c_0\f$,
   * \f$c_1\f$, \f$c_2\f$, and \f$c_3\f$ are the fitting constants.
   * 
   * @param rTime The maturities of market discount factors.
   * @param rDF The market discount factors. 
   * @param dLambda1 (\f$\lambda_1 \geq 0\f$) The first mean-reversion rate.
   * @param dLambda2 (\f$\lambda_2 \geq 0\f$) The second
   * mean-reversion rate, \f$\lambda_2\not=\lambda_1\f$.
   * @param dInitialTime (\f$t_0\f$) The initial time.
   * @param rErr The error function of the fit for discount factors. 
   * @param rParam The fitted coefficients, their covariance matrix, and
   * the total \f$\chi^2\f$ fitting error.
   * 
   * @return The fitted discount curve and its error function. We also
   * obtain the fitted coefficients and their covariance matrix.
   */
  cfl::Function
  discountSvenssonFit(const std::vector<double> &rTime,
                      const std::vector<double> &rDF,
                      double dLambda1, double dLambda2,
                      double dInitialTime, cfl::Function &rErr,
		      cfl::FitParam & rParam);
  /** @} */  
}

#endif // of __Homework2_hpp__
