#ifndef __Exam_hpp__
#define __Exam_hpp__

/**
 * @file Exam.hpp
 * @author Dmitry Kramkov (kramkov@andrew.cmu.edu)
 * @brief Exam for the course. 
 * @version 0.1
 * @date 2021-02-12
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "cfl/Data.hpp"
#include "cfl/Interp.hpp"
#include "cfl/AssetModel.hpp"
#include "cfl/InterestRateModel.hpp"

/**
 * @mainpage Exam for the course "Financial Computing with C++"
 */

namespace prb
{
  /** 
   * @defgroup prbData Data structures. 
   *
   * This module constructs input data curves for financial models. 
   * @{
   */

  /** 
   * Computes the curve of forward prices for a cash flow.  The buyer
   * pays forward price \f$F(t)\f$ at delivery time \f$t\f$ and then
   * receives payments \f$(P_i)\f$ at payments times \f$t_i>t\f$.
   * 
   * @param rPayments \f$(P_i)\f$ The vector of payments. 

   * @param rPaymentTimes \f$(t_i)\f$ The vector of payment times.
   * The first payment time has to be strictly greater than the
   * initial time.
   * @param rDiscount The discount curve. 
   * @param dInitialTime (\f$t_0\f$) The initial time. 
   * 
   * @return The forward prices \f$(F(t))_{t\in [t_0,T]}\f$ for the cash
   * flow, where \f$T\f$ is the last payment time.
   */
  cfl::Function forwardCashFlow(const std::vector<double> &rPayments,
                                const std::vector<double> &rPaymentTimes,
                                const cfl::Function &rDiscount,
                                double dInitialTime);
  /** @} */

  /** 
   * @defgroup prbInterp Interpolation of data curves. 
   *
   * This module constructs input data curves for financial 
   * models using interpolation.   
   * @{
   */

  /**
   * Computes forward curve 
   * \f[
   * F(t) = S_0 e^{q(t)(t-t_0)}, \quad t\in [t_0,t_M], 
   * \f]
   * where cost-of-carry rate \f$q=q(t)\f$ is obtained by the linear
   * interpolation to the market cost-of-carry rates. On the first
   * interval \f$[t_0,t_1]\f$, cost-of-carry rate function
   * \f$q=q(t)\f$ is constant.
   * 
   * @param dSpot (\f$S_0\f$) The spot price of the stock. 
   * @param rDeliveryTimes \f$(t_i)_{i=1,\dots,M}\f$ The delivery
   * times for the market forward prices.
   * @param rForwardPrices The market forward prices.  
   * @param dInitialTime (\f$t_0\f$) The initial time.
   * 
   * @return The forward curve obtained by the linear interpolation of
   * the market cost-of-carry rates.
   */
  cfl::Function forwardCarryLinInterp(double dSpot,
                                      const std::vector<double> &rDeliveryTimes,
                                      const std::vector<double> &rForwardPrices,
                                      double dInitialTime);
  /** @} */

  /**
   * @defgroup cflAssetOptions Options on a single stock. 
   * 
   * This module deals with valuation of options on a single stock.
   * @{
   */
  /** 
   * At maturity \f$T\f$ (\p dMaturity) the holder gets payment
   * \f[
   * V(T) = N\left(1 + q\max\left(\frac{S(T)}K -1,0 \right)\right),
   * \f]
   * if the buy-back provision has not been used. Here \f$N\f$ 
   * is the notional (\p dNotional), \f$q\f$ is the participation rate (\p dQ), 
   * \f$S(T)\f$ is the stock price at the maturity and \f$K\f$ is the strike
   * (\p dStrike). The buy-back feature allows the holder to sell the note 
   * back at one of the buy-back times (\p rBuyBackTimes) for the 
   * amount <code>dP*dNotional</code>.
   * 
   * @param dNotional The notional amount.
   * @param dStrike The strike.
   * @param dQ The participation rate. 
   * @param dP The buy-back percentage of the notional. 
   * @param rBuyBackTimes The buy-back times. 
   * @param dMaturity The maturity. 
   * @param rModel The reference to an implementation of cfl::AssetModel. 
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model. 
   */
  cfl::MultiFunction
  equityLinkedNoteWithBuyBack(double dNotional, double dStrike,
                              double dQ, double dP,
                              const std::vector<double> &rBuyBackTimes,
                              double dMaturity, cfl::AssetModel &rModel);
  /** @} */

  /**
   * @defgroup cflInterestRateOptions Options on interest rates. 
   * 
   * This module deals with valuation of standard and barrier 
   * options on interest rates.
   * 
   * @{
   */
  /** 
   * Computes the price of <strong>American constant maturity
   * swaption</strong>.  A holder of the option can enter into the
   * underlying CMS (constant maturity swap) agreement at any
   * exercise time. This time then becomes the issue time of CMS. In
   * the CMS the float interest is paid according to the market
   * swap rate in the swap contract expiring after \p iCMSPeriods.
   * 
   * @param rSwap The parameters of the underlying interest rate
   * swap. 
   * @param rExerciseTimes The set of exercise times. The first
   * exercise time is strictly greater than the initial time in the
   * model. 
   * @param iCMSPeriods The number of periods in the swap contract 
   * that determines the float rate. 
   * @param rModel The reference to the implementation of cfl::InterestRateModel.
   * 
   * @return The price of the option as the function of the initial
   * values of the state processes in the model.
   */
  cfl::MultiFunction americanCMSwaption(const cfl::Data::Swap &rSwap,
                                        const std::vector<double> &rExerciseTimes,
                                        unsigned iCMSPeriods,
                                        cfl::InterestRateModel &rModel);
  /** @} */
} // namespace prb

#endif // of __Exam_hpp__
