#include "test/Main.hpp"
#include "test/Data.hpp"
#include "test/Print.hpp"
#include "test/Black.hpp"
#include "test/HullWhite.hpp"
#include "Exam/Output.hpp"
#include "Exam/Exam.hpp"

using namespace test;
using namespace cfl;
using namespace std;
using namespace test::Data;

void forwardCashFlow()
{
  test::print("FORWARD PRICES FOR A CASH FLOW");

  double dRate = test::HullWhite::c_dYield;
  double dInitialTime = 1.;
  unsigned iPayments = 6;
  std::vector<double> uPayments(iPayments);
  uPayments.front() = 100.;
  std::transform(uPayments.begin(), uPayments.end() - 1, uPayments.begin() + 1,
                 [](double dX) {
                   return dX * (1. + 0.5 * std::cos(dX));
                 });
  std::vector<double> uTimes =
    getTimes(dInitialTime + 0.5, dInitialTime + 3., iPayments);
  Function uDiscount = cfl::Data::discount(dRate, dInitialTime);
  test::print(dRate, "interest rate");
  test::print(dInitialTime, "initial time", true);
  test::Data::print("cash flow:", uTimes, uPayments);
  Function uForwardCashFlow =
    prb::forwardCashFlow(uPayments, uTimes, uDiscount,
			 dInitialTime);
  double dInterval = (uTimes.back() - dInitialTime) / 1.01;
  test::Data::print(uForwardCashFlow, dInitialTime, dInterval);
}

void forwardCarryLinInterp()
{
  test::print("FORWARD PRICES  BY LINEAR INTERPOLATION OF COST-OF-CARRY RATES");

  double dSpot = 100;
  double dInitialTime = 1.;

  auto uF = test::Data::getForward(dSpot, dInitialTime);

  Function uResult = prb::forwardCarryLinInterp(dSpot, uF.first,
                                                uF.second, dInitialTime);

  double dInterval = uF.first.back() - dInitialTime+0.25;
  test::Data::print(uResult, dInitialTime, dInterval);
}

MultiFunction equityLinkedNoteWithBuyBack(AssetModel &rModel)
{
  test::print("EQUITY LINKED NOTE WITH BUY-BACK PROVISION IN ASSET MODEL");

  double dNotional = test::Black::c_dNotional;
  double dStrike = test::Black::c_dSpot;
  double dQ = 0.5;
  double dP = 0.9;
  const std::vector<double> uBuyBackTimes = test::Black::barrierTimes();
  double dMaturity = uBuyBackTimes.back() + 0.05;

  print(dNotional, "notional");
  print(dStrike, "strike");
  print(dQ, "participation rate");
  print(dP, "buy-back percentage");
  print(dMaturity, "maturity", true);
  test::print(uBuyBackTimes.begin(), uBuyBackTimes.end(), "buy-back times");

  return prb::equityLinkedNoteWithBuyBack(dNotional, dStrike,
                                          dQ, dP, uBuyBackTimes,
                                          dMaturity, rModel);
}

cfl::MultiFunction americanCMSwaption(InterestRateModel &rModel, bool bPayFloat)
{
  test::print("AMERICAN CONSTANT MATURITY SWAPTION IN INTEREST RATE MODEL");

  cfl::Data::Swap uSwap =
    test::HullWhite::swapParameters();
  uSwap.payFloat = bPayFloat;
  std::vector<double> uExerciseTimes = test::HullWhite::exerciseTimes();
  unsigned iCMSPeriods = 3;

  print(iCMSPeriods, "CMS periods", true);
  test::print(uExerciseTimes.begin(), uExerciseTimes.end(), "exercise  times");
  test::printSwap(uSwap, "swap parameters");

  return prb::
    americanCMSwaption(uSwap, uExerciseTimes, iCMSPeriods, rModel);
}

std::function<void()> test_Exam()
{
  return []() {
	   print("DATA CURVES FOR FINANCIAL MODELS");

	   forwardCashFlow();

	   print("INTERPOLATION OF DATA CURVES");

	   forwardCarryLinInterp();

	   print("OPTIONS ON A SINGLE STOCK IN BLACK MODEL");

	   AssetModel uBlack = test::Black::model();
	   test::Black::report(equityLinkedNoteWithBuyBack,uBlack);

	   print("INTEREST RATE OPTIONS IN HULL-WHITE MODEL");

	   InterestRateModel uHullWhite = test::HullWhite::model();
	   test::HullWhite::report(americanCMSwaption, uHullWhite);
	 };
}

int main()
{
  project(test_Exam(), PROJECT_NAME, PROJECT_NAME,
          "Exam");
}
