# Use like:
# python extended.py 48 35
# or
# python extended.py 7 3

import numbers

def egcd(a, b):
    """
    Implementation of the extended Euclidean algorithm
    for computing
     - the greatest common divisor of a and b
     - the coefficients satisfying
       ax + by = gcd(a, b)
    """
    assert a >= b
    assert b > 0
    assert isinstance(b, numbers.Integral)
    assert isinstance(b, numbers.Integral)

    (r_prev, r_next) = (a, b)
    (x_prev, y_prev) = (1, 0)
    (x_next, y_next) = (0, 1)

    while r_next > 0:
        q = r_prev // r_next
        (r_prev, r_next) = (r_next, r_prev - q*r_next)
        (x_prev, x_next) = (x_next, x_prev - q*x_next)
        (y_prev, y_next) = (y_next, y_prev - q*y_next)
        assert a*x_next + b*y_next == r_next

    return (r_prev, x_prev, y_prev)

if __name__ == "__main__":
    import sys

    (a, b) = int(sys.argv[1]), int(sys.argv[2])
    (gcd, x, y) = egcd(a, b)
    print(f"{a} * {x} + {b} * {y} == {gcd}")
    assert a*x + b*y == gcd
