import mpmath
mpmath.mp.dps = 100

from mpmath import log, pi, mpf, fabs
from itertools import count

def leibniz():
  # Use a Python generator to model the infinite
  # sequence of partial sums
  out = 0
  for n in count():
    out += (-1)**n / mpf(2*n+1)
    yield 4*out

def aitken(seq):
  values = (next(seq), next(seq), next(seq))
  while True:
    ahat = values[0] - (values[1] - values[0])**2/(values[2] -2*values[1] + values[0])
    yield ahat
    values = (values[1], values[2], next(seq))

from itertools import islice

N = 10003

base = list(fabs(x - pi) for x in islice(leibniz(), 0, N))
once = list(fabs(x - pi) for x in islice(aitken(leibniz()), 0, N))
twice = list(fabs(x - pi) for x in islice(aitken(aitken(leibniz())), 0, N))
thrice = list(fabs(x - pi) for x in islice(aitken(aitken(aitken(leibniz()))), 0, N))

print(f"Unaccelerated sequence: error at {N=}: ", base[-1])
print(f"Aitken applied once:    error at {N=}: ", once[-1], "ratio: ", base[-1]/once[-1])
print(f"Aitken applied twice:   error at {N=}: ", twice[-1], "ratio: ", once[-1]/twice[-1])
print(f"Aitken applied thrice:  error at {N=}: ", thrice[-1], "ratio: ", twice[-1]/thrice[-1])

import matplotlib.pyplot as plt

seabornblue  = (0.2980392156862745, 0.4470588235294118, 0.6901960784313725)
seabornred   = (0.7686274509803922, 0.3058823529411765, 0.3215686274509804)
seaborngreen = (0.3333333333333333, 0.6588235294117647, 0.40784313725490196)

xcoord = list(range(N))
plt.semilogy(xcoord, base, "-k", label="leibniz")
plt.semilogy(xcoord, once, "-", color=seabornred, label="accelerated once")
plt.semilogy(xcoord, twice, "-", color=seabornblue, label="accelerated twice")
#plt.semilogy(xcoord, thrice, "-", color=seaborngreen, label="accelerated thrice")
plt.xlabel(r"$n$")
plt.ylabel(r"$|a_n - \pi|$")
plt.legend()
plt.show()

# Estimating order of convergence
# This assumes that the error at iteration $e_k$ satisfies
# e_{k+1} = C · e_k^p
# and compares logarithms of ratios of successive terms to estimate p.
# Once p is estimated, we can estimate C.

def convergence_stats(errors):
    q = log(errors[-1]/errors[-2]) / log(errors[-2]/errors[-3])
    M = errors[-1]/errors[-2]**q
    return (q, M)

(q, M) = convergence_stats(base)
print(f"Convergence stats for unaccelerated sequence:      e_(n+1) = {float(M):.4e} * e_n^({float(q):.4e})")
(q, M) = convergence_stats(once)
print(f"Convergence stats for sequence accelerated once:   e_(n+1) = {float(M):.4e} * e_n^({float(q):.4e})")
(q, M) = convergence_stats(twice)
print(f"Convergence stats for sequence accelerated twice:  e_(n+1) = {float(M):.4e} * e_n^({float(q):.4e})")
(q, M) = convergence_stats(thrice)
print(f"Convergence stats for sequence accelerated thrice: e_(n+1) = {float(M):.4e} * e_n^({float(q):.4e})")
