import numbers
import math

def extended_euclid(a, b, c):
    if a < b:
        # swap around
        out = extended_euclid(b, a, c)
        return (out[1], out[0])
    assert b > 0
    assert isinstance(b, numbers.Integral)
    assert isinstance(b, numbers.Integral)

    # Compute GCD to check if solutions exist
    g = math.gcd(a, b)
    if c % g != 0:
        raise ValueError("No solutions")

    # Construct reduced problem
    a = a//g
    b = b//g
    c = c//g

    (r_prev, r_next) = (a, b)
    (x_prev, y_prev) = (1, 0)
    (x_next, y_next) = (0, 1)

    # Iterate
    while r_next > 0:
        q = r_prev // r_next
        (r_prev, r_next) = (r_next, r_prev - q*r_next)
        (x_prev, x_next) = (x_next, x_prev - q*x_next)
        (y_prev, y_next) = (y_next, y_prev - q*y_next)
        assert a*x_next + b*y_next == r_next

    return (c*x_prev, c*y_prev)

if __name__ == "__main__":
    import sys

    try:
        (a, b, c) = int(sys.argv[1]), int(sys.argv[2]), int(sys.argv[3])
    except IndexError:
        print(f"Use like: {sys.argv[0]} a b c")
        print( "to solve ax + by = c.")
        sys.exit(1)
    try:
        (x, y) = extended_euclid(a, b, c)
        print(f"{a} * {x} + {b} * {y} == {c}")
        assert a*x + b*y == c
    except ValueError:
        print(f"No solutions.")
