close all
clear all

% Solving free boundary problem (PDE system) using method of lines
% G.P. Benham
% Oxford, 10/10/2022

% Discretise domain
N = 200;
xmax=2;
x = linspace(0,xmax,N)';
dx = x(2);

% Choose order of accuracy of finite difference matrix (even number)
ooa=2;

% Make finite difference matrix
Dx = getNonCompactFDmatrix(N,dx,1,ooa);

% Total simulation time
tstop=1;

% Time step
dt=1e-3;

% Set inital conditions as a small triangular shape (with compact support)
x0=0.1;
H=x0-x;
H(H<0)=0;

% Commence a 'for' loop, calculating soluition at every time step
t=0;i=0;Hnew=H;
while t<tstop 
    t=t+dt;i=i+1;
    
    % Create nonlinear system of equations (see bottom of script)
    odefun=@(solnew) impsol(solnew,Hnew,Dx,dt);
    
    % Solve system of nonlinear equations using Newton's method, and using previous solution as initial guess
    Hnew=fsolve(odefun,Hnew,optimset('display','off'));
    
    % Update solution
    H=Hnew;
    
    % Plot the solution every 100 time steps
    if mod(i,100)==0
        figure(1);clf;
        plot(x,H,'linewidth',2)
        title(strcat('$$t=',num2str(round(t,2)),'$$'),'interpreter','latex')
        ylim([0,xmax])
        xlim([0,xmax])
        set(gca,'fontsize',20)
        xlabel('$$x$$','interpreter','latex')
        ylabel('$$z$$','interpreter','latex')
        grid on
        pause(0.01)
    end
end

% Function creating the nonlinear PDE equations
function G = impsol(H,Hold,Dx,dt)

% Create derivatives of H
Hx=Dx*H;
flux=H.*Hx;
divH=Dx*flux;

% PDE system, discretised in time using first order Euler
G=H-Hold-dt*divH;

% First equation set as the left hand BC
G(1)=flux(1)+1;

end
