close all
clear all

% Solving free boundary problem using all-at-once approach
% G.P. Benham
% Oxford, 10/10/2022

% Discretise
N=1000;
x=linspace(0,1,N)';
dx=x(2);

% Choose order of accuracy of finite difference matrix (even number)
ooa=2;

% Make finite difference matrix
Dx = getNonCompactFDmatrix(N,dx,1,ooa);

% Set initial guess for variables
f0=1-x;
etaN=1;
y0=[f0;etaN];

% Create nonlinear system of equations (see bottom of script)
F=@(y) nonlinsys(y,Dx,x);

% Solve system of nonlinear equations using Newton's method: N.B. Jacobian is unknown, so Newton will approximate using function evaluations
y=fsolve(F,y0); 

% Get the solution
f=y(1:end-1);
etaN=y(end)

% Plot the solution
plot(x,f,'b-','linewidth',2)
grid on
box on
set(gca,'fontsize',20)
xlabel('$$\eta/\eta_N$$','interpreter','latex')
ylabel('$$f$$','interpreter','latex')


% Function creating the nonlinear system of equations
function G = nonlinsys(y,Dx,x)

% Decision variables
f=y(1:end-1);
etaN=y(end);

% Create derivatives 
fdash=Dx*f;
fddash=Dx*fdash;

% Nonlinear system of equations
% ODE equations
G=etaN^2/3*(f-2*x.*fdash)-(fdash.^2+f.*fddash);

% First equation (left hand BC)
G(1)=fdash(1)*f(1)+etaN;

% Penultimate equation (right hand BC)
G(end)=f(end);

% Last equation (integral constraint)
int=etaN*trapz(x,f)-1;

G=[G;int];
end
