close all
clear all

% Solving free boundary problem using shooting method 
% G.P. Benham
% Oxford, 10/10/2022

% Initial guess for etaN
etaN=1;

% Number of discretisation points
N=1000;

% Construct nonlinear equation (integral constraint, see bottom of script)
f=@(etaN) nonlinfun(etaN,N);

% Find root of nonlinear equation using Newton's method
etaN=fsolve(f,etaN)

% Use the correct value of etaN to now calculate function f
[~,f] = nonlinfun(etaN,N);

% Plot the solution
figure(2);
y=linspace(0,1,N);
plot(y,f,'k-','linewidth',2)
box on
grid on
set(gca,'fontsize',20)
xlabel('$$\eta/\eta_N$$','interpreter','latex')
ylabel('$$f$$','interpreter','latex')
hold on

% Function creating the nonlinear equation (integral constraint)
function [obj,fs] = nonlinfun(etaN,N)

% Discretisation
y=linspace(0,1,N);
dy=y(2);

% Set variables at the right hand boundary y=1 (no flux, L=0 and f=0)
yi=y(end);
L=0;
f=0;

% Calculate initial right hand side of first order ODE system
fdash=-2*etaN^2/3;
Ldash=-etaN^2/3*( f -2*yi*fdash);

% Using a 'for' loop, calculate points from right to left
fs=[f];
for i=1:N-1
    % Set current y value
    yi=y(N-i);
    
    % Calculate next values on the left for f and L (1st order scheme)
    f=f-dy*fdash;
    L=L-dy*Ldash;
    
    % Update right hand side of first order ODE system
    fdash=-L/f;
    Ldash=-etaN^2/3*( f -2*yi*fdash);

    % Create vector for solution
    fs=[fs,f];
end

% Calculate integral value using trapezoidal rule
fs=flip(fs);
intval=etaN*trapz(y,fs);

% Set output of the nonlinear function
obj=intval-1;

end